/**
 *
 */
package multisab.processing.dataHandling;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;


/**
 * @author Davor
 */
public class EdfFile extends InputData {

    // TODO calculate appropriate buffer size depending on dataRecord size?
    private final int BUFFER_SIZE_BYTES = 524288; //0.5 megabytes 524288
    private ArrayList<Integer> indexOfEDFAnnotations;
    private ArrayList<Integer> indexOfSignals;

    public EdfFile(File selectedFile, boolean loadHeaderOnly) throws IOException {

        super(selectedFile);

        BufferedInputStream in = null;

        if (recordType.equals("EDF")) {
            if (!loadHeaderOnly) {
                try {
                    in = new BufferedInputStream(new FileInputStream(selectedFile));

                    indexOfEDFAnnotations = new ArrayList<Integer>();
                    indexOfSignals = new ArrayList<Integer>();

                    loadEdfHeader(in);
                    loadEdfData(in);

                } finally {
                    if (in != null) {
                        in.close();
                    }
                }

                findMinAndMaxSampleAllSignals();
            }
            else {
               try {
                     in = new BufferedInputStream(new FileInputStream(selectedFile));

                     indexOfEDFAnnotations = new ArrayList<Integer>();
                     indexOfSignals = new ArrayList<Integer>();

                     loadEdfHeader(in);

               } finally {
                   if (in != null) {
                       in.close();
                   }
               }
            }
        }

    }


    private void loadEdfHeader(BufferedInputStream input) throws IOException { //throws new FormatCorruptedException
        // TODO check EDF/EDF+ compatibility (max digital sample and samples in dataRecords)
        metadata = new Metadata();

        String versionOfDataFormat = readBytesToString(input, EDFHeaderBytesNum.VERSION_OF_DATA_FORMAT.getValue());
        versionOfDataFormat = versionOfDataFormat.trim();
        metadata.setVersionOfDataFormat(versionOfDataFormat);

        String localPatientId = readBytesToString(input, EDFHeaderBytesNum.LOCAL_PATIENT_IDENTIFICATION.getValue());
        localPatientId = localPatientId.trim();
        metadata.setLocalPatientIdentification(localPatientId);

        String localRecordingId = readBytesToString(input, EDFHeaderBytesNum.LOCAL_RECORDING_IDENTIFICATION.getValue());
        localRecordingId = localRecordingId.trim();
        metadata.setLocalRecordingIdentification(localRecordingId);

        String startDate = readBytesToString(input, EDFHeaderBytesNum.RECORDING_START_DATE.getValue());
        startDate = startDate.trim();
        metadata.setRecordingStartDate(startDate);

        String startTime = readBytesToString(input, EDFHeaderBytesNum.RECORDING_START_TIME.getValue());
        startTime = startTime.trim();
        metadata.setRecordingStartTime(startTime);

        String headerLength = readBytesToString(input, EDFHeaderBytesNum.HEADER_LENGTH.getValue());
        headerLength = headerLength.trim();
        metadata.setHeaderLengthInBytes(Long.parseLong(headerLength));

        String reserved = readBytesToString(input, EDFHeaderBytesNum.RESERVED.getValue());
        reserved = reserved.trim();
        metadata.setReserved(reserved);
        if (reserved.equals("EDF+C")) {
            recordType = "EDF+C";
        }
        if (reserved.equals("EDF+D")) {
            recordType = "EDF+D";
        }
        String dataRecordsNum = readBytesToString(input, EDFHeaderBytesNum.DATA_RECORDS_NUM.getValue());
        dataRecordsNum = dataRecordsNum.trim();
        metadata.setDataRecordsNum(Long.parseLong(dataRecordsNum));
        if (metadata.getDataRecordsNum() > Integer.MAX_VALUE) {
            // TODO Handle large files, read as two files, or print message and close
            //throw new TooManyDataRecordsException
        }

        String dataRecordDuration = readBytesToString(input, EDFHeaderBytesNum.DATA_RECORD_DURATION.getValue());
        dataRecordDuration = dataRecordDuration.trim();
        metadata.setDataRecordDurationInSec(Double.parseDouble(dataRecordDuration));

        String signalNum = readBytesToString(input, EDFHeaderBytesNum.SIGNALS_NUM.getValue());
        signalNum = signalNum.trim();
        metadata.setSignalsNum(Integer.parseInt(signalNum));
        numberOfSignals = metadata.getSignalsNum();

        loadParameters(input);

        duration = metadata.getDataRecordsNum() * metadata.getDataRecordDurationInSec();
    }

    private void loadParameters(BufferedInputStream input) throws IOException {

        if (metadata.getSignalsNum() <= 0) {
            //throw new ErrorInFormatException
            //exit
        }
        SignalParameterData[] signals = new SignalParameterData[metadata.getSignalsNum()];

        loadLabels(input, signals);
        loadTransducerTypes(input, signals);
        loadPhysicalDimensions(input, signals);
        loadPhysicalMins(input, signals);
        loadPhysicalMaxs(input, signals);
        loadDigitalMins(input, signals);
        loadDigitalMaxs(input, signals);
        loadPrefilterings(input, signals);
        loadSamplesInDataRecordNums(input, signals);
        loadReserveds(input, signals);

        metadata.setSignalParameters(signals);
    }

    private void loadLabels(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            SignalParameterData signal = new SignalParameterData();
            String label = readBytesToString(input, EDFSignalParametersBytesNum.LABEL.getValue());
            label = label.trim();
            signal.setLabel(label);
            if (label.equals("EDF Annotations")) {
                this.numberOfSignals = this.numberOfSignals - 1;
                this.numberOfAnnotations = this.numberOfAnnotations + 1;
                indexOfEDFAnnotations.add(sigNum);
                signal.setType(1);
            } else {
                indexOfSignals.add(sigNum);
                signal.setType(0);
            }
            signals[sigNum] = signal;

        }
    }

    private void loadTransducerTypes(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            String transducer = readBytesToString(input, EDFSignalParametersBytesNum.TRANSDUCER_TYPE.getValue());
            transducer = transducer.trim();
            signals[sigNum].setTransducerType(transducer);
        }
    }

    private void loadPhysicalDimensions(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            String physicalDimension = readBytesToString(input, EDFSignalParametersBytesNum.PHYSICAL_DIMENSION.getValue());
            physicalDimension = physicalDimension.trim();
            signals[sigNum].setPhysicalDimension(physicalDimension);
        }
    }

    private void loadPhysicalMins(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            String physicalMin = readBytesToString(input, EDFSignalParametersBytesNum.PHYSICAL_MIN.getValue());
            physicalMin = physicalMin.trim();
            signals[sigNum].setPhysicalMin(Double.parseDouble(physicalMin));
        }
    }

    private void loadPhysicalMaxs(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            String physicalMax = readBytesToString(input, EDFSignalParametersBytesNum.PHYSICAL_MAX.getValue());
            physicalMax = physicalMax.trim();
            signals[sigNum].setPhysicalMax(Double.parseDouble(physicalMax));
        }
    }

    private void loadDigitalMins(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            String digitalMin = readBytesToString(input, EDFSignalParametersBytesNum.DIGITAL_MIN.getValue());
            digitalMin = digitalMin.trim();
            signals[sigNum].setDigitalMin(Double.parseDouble(digitalMin));
        }
    }

    private void loadDigitalMaxs(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            String digitalMax = readBytesToString(input, EDFSignalParametersBytesNum.DIGITAL_MAX.getValue());
            digitalMax = digitalMax.trim();
            signals[sigNum].setDigitalMax(Double.parseDouble(digitalMax));
        }
    }

    private void loadPrefilterings(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            String prefiltering = readBytesToString(input, EDFSignalParametersBytesNum.PREFILTERING.getValue());
            prefiltering = prefiltering.trim();
            signals[sigNum].setPrefiltering(prefiltering);
        }
    }

    private void loadSamplesInDataRecordNums(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            String samplesNum = readBytesToString(input, EDFSignalParametersBytesNum.SAMPLES_NUM_IN_DATA_RECORD.getValue());
            samplesNum = samplesNum.trim();
            signals[sigNum].setSamplesInDataRecordNum(Long.parseLong(samplesNum));
        }
    }

    private void loadReserveds(BufferedInputStream input, SignalParameterData[] signals) throws IOException {
        for (int sigNum = 0; sigNum < signals.length; sigNum++) {
            String reserved = readBytesToString(input, EDFSignalParametersBytesNum.RESERVED.getValue());
            reserved = reserved.trim();
            signals[sigNum].setReserved(reserved);
        }
    }

    private void loadEdfData(BufferedInputStream input) throws IOException {
        // TODO  dataRecordNum compare to metadata.dataRecordsNum value before calling this method
        //calculate correct number of dataRecords if -1
        //throw new TooManyDataRecordsException

        int signalsNum = metadata.getSignalsNum();

        DataRecord[] dataRecords;

        short[][] data;
        dataRecords = new DataRecord[(int) metadata.getDataRecordsNum()];
        for (int k = 0; k < dataRecords.length; k++) {
            dataRecords[k] = new DataRecord();

            data = new short[signalsNum][];
            for (int sig = 0; sig < signalsNum; sig++) {
                data[sig] = new short[(int) metadata.getSignalParameters()[sig].getSamplesInDataRecordNum()];
            }
            dataRecords[k].setSignalData(data);
        }

        byte[] b = new byte[BUFFER_SIZE_BYTES], pom;
        short[] shorts;
        int i = 0, j = 0, dataRecordsNum = 0, length = 0, index = 0;
        long samplesLeft = metadata.getSignalParameters()[i].getSamplesInDataRecordNum();
        int numByteRead = -1;
        while ((numByteRead = input.read(b, 0, b.length)) > -1) {

            pom = new byte[numByteRead];
            System.arraycopy(b, 0, pom, 0, numByteRead);

            shorts = convertBytesToShorts(pom);

            index = 0;
            while (true) {

                if (samplesLeft > Integer.MAX_VALUE) { //TODO handle long number of samples
                    length = shorts.length;
                } else {
                    length = Math.min(shorts.length - index, (int) samplesLeft);
                }

                System.arraycopy(shorts, index, dataRecords[dataRecordsNum].getSignalData()[i], j, length);
                //	    		if(dataRecordsNum == 0 && i == 0){
                //		        	  for(int bla = 0; bla < length; bla++){
                //		  	        	System.out.println("ovo su shorts elementi "+shorts[bla] + " a ovo odgovarajući dataRecords elementi "+ dataRecords[dataRecordsNum].getSignalData()[i][bla]);
                //
                //		  	        }
                //		        }


                samplesLeft -= length;
                j += length;
                index += length;

                if (samplesLeft == 0) {
                    j = 0;
                    if (i < signalsNum - 1) {
                        i++;
                        samplesLeft = metadata.getSignalParameters()[i].getSamplesInDataRecordNum();
                    } else {
                        dataRecordsNum++;
                        i = 0;
                        samplesLeft = metadata.getSignalParameters()[i].getSamplesInDataRecordNum();
                    }
                }
                if (index == shorts.length) {
                    break;
                }
            }
        }

        signals = new double[this.numberOfSignals][];
        this.numberOfSamples = new int[this.numberOfSignals];

        for (i = 0; i < indexOfSignals.size(); i++) {

            int sigIndex = indexOfSignals.get(i);

            int numSamplesInDataRecord = (int) metadata.getSignalParameters()[sigIndex].getSamplesInDataRecordNum();
            int numDataRecords = (int) metadata.getDataRecordsNum();
            int numSamplesInLastDataRecord = dataRecords[(int) (metadata.getDataRecordsNum() - 1)].getSignalData()[sigIndex].length;

            this.numberOfSamples[i] = numSamplesInDataRecord * (numDataRecords - 1) + numSamplesInLastDataRecord;

            double[] signal = new double[this.numberOfSamples[i]];

            int maxDigital = (int) metadata.getSignalParameters()[sigIndex].getDigitalMax();
            int minDigital = (int) metadata.getSignalParameters()[sigIndex].getDigitalMin();
            double maxPhysical = (int) metadata.getSignalParameters()[sigIndex].getPhysicalMax();
            double minPhysical = (int) metadata.getSignalParameters()[sigIndex].getPhysicalMin();

            double coef = (maxPhysical - minPhysical) / (maxDigital - minDigital);

            for (j = 0; j < (numDataRecords - 1); j++) {
                //Za short //System.arraycopy(dataRecords[j].getSignalData()[sigIndex], 0, signal, j*numSamplesInDataRecord, numSamplesInDataRecord);
                for (int k = 0; k < numSamplesInDataRecord; k++) {
                    signal[j * numSamplesInDataRecord + k] = minPhysical + (dataRecords[j].getSignalData()[sigIndex][k] - minDigital) * coef;
                }
            }
            //Za short //System.arraycopy(dataRecords[j].getSignalData()[sigIndex], 0, signal, j*numSamplesInDataRecord, numSamplesInLastDataRecord);
            for (int k = 0; k < numSamplesInLastDataRecord; k++) {
                signal[j * numSamplesInDataRecord + k] = minPhysical + (dataRecords[j].getSignalData()[sigIndex][k] - minDigital) * coef;
            }

            signals[i] = signal;
        }


        /**
         * Number of annotations in the signal - it is set to 0 if the signal is the whole signal
         */
        int annotationsCount = 0;
        /**
         * List of annotations
         */
        ArrayList<Annotation> annotsList;

        annotsList = new ArrayList<Annotation>();

        short c;
        short c1;
        short c2;

        annots = new Annotation[indexOfEDFAnnotations.size()][];
        numberOfAnnotsSamples = new int[indexOfEDFAnnotations.size()];

        for (i = 0; i < indexOfEDFAnnotations.size(); i++) {

            int sigIndex = indexOfEDFAnnotations.get(i);

            int numSamplesInDataRecord = 2 * (int) metadata.getSignalParameters()[sigIndex].getSamplesInDataRecordNum();
            int numDataRecords = (int) metadata.getDataRecordsNum();

            for (j = 0; j < numDataRecords; j++) {

                short[] signal = new short[numSamplesInDataRecord];

                for (int k = 0; k < numSamplesInDataRecord / 2; k++) {
                    c1 = (short) (dataRecords[j].getSignalData()[sigIndex][k] % 256);
                    c2 = (short) (dataRecords[j].getSignalData()[sigIndex][k] / 256);
                    if (c1 == 0 && c2 == 0) {
                        break;
                    }
                    signal[k * 2] = c1;
                    signal[k * 2 + 1] = c2;
                }

                String dataRecordOnset = "";
                int k = 0;
                int numberOfOnsetMarker = 0;
                int numberOfDurationMarker = 0;
                while (numberOfOnsetMarker < 2) {
                    c = signal[k];
                    k++;

                    if (c == 20) {
                        numberOfOnsetMarker++;
                    } else if (numberOfOnsetMarker < 1) {
                        dataRecordOnset = dataRecordOnset + (char) c;
                    }
                }


                while (!((signal[k] == 0) && (signal[k + 1] == 0))) {

                    String duration = "";
                    String onset = "";
                    String event = "";

                    numberOfOnsetMarker = 0;
                    numberOfDurationMarker = 0;
                    while (numberOfOnsetMarker < 2) {
                        c = signal[k];
                        k++;

                        if (c == 20) {
                            numberOfOnsetMarker++;
                        } else if (c == 21) {
                            numberOfDurationMarker++;
                        } else if ((numberOfOnsetMarker < 1) && (numberOfDurationMarker < 1)) {
                            onset = onset + (char) c;
                        } else if ((numberOfOnsetMarker == 0) && (numberOfDurationMarker == 1)) {
                            duration = duration + (char) c;
                        } else if (numberOfOnsetMarker == 1) {
                            event = event + (char) c;
                        }
                    }

                    Annotation ann = new Annotation();

                    ann.timeAnnots = Double.parseDouble(onset);
                    if (numberOfDurationMarker == 1) {
                        ann.duration = Double.parseDouble(duration);
                    } else {
                        ann.duration = 0;
                    }
                    ann.aux = "" + event;
                    annotationsCount++;
                    annotsList.add(ann);
                }
            }

            Annotation[] signalAnnots = new Annotation[annotationsCount];

            for (int k = 0; k < annotsList.size(); k++) {
                signalAnnots[k] = annotsList.get(k);
            }

            annots[i] = signalAnnots;
            numberOfAnnotsSamples[i] = annotationsCount;
        }
    }

    @Override
    public double calculateFrequency(int signalIndex) {
        return signals[signalIndex].length / duration;
    }


    private enum EDFHeaderBytesNum {

        VERSION_OF_DATA_FORMAT(8),
        LOCAL_PATIENT_IDENTIFICATION(80),
        LOCAL_RECORDING_IDENTIFICATION(80),
        RECORDING_START_DATE(8),
        RECORDING_START_TIME(8),
        HEADER_LENGTH(8),
        RESERVED(44),
        DATA_RECORDS_NUM(8),
        DATA_RECORD_DURATION(8),
        SIGNALS_NUM(4);

        private final int id;

        EDFHeaderBytesNum(int id) {
            this.id = id;
        }

        public int getValue() {
            return id;
        }
    }

    private enum EDFSignalParametersBytesNum {
        LABEL(16),
        TRANSDUCER_TYPE(80),
        PHYSICAL_DIMENSION(8),
        PHYSICAL_MIN(8),
        PHYSICAL_MAX(8),
        DIGITAL_MIN(8),
        DIGITAL_MAX(8),
        PREFILTERING(80),
        SAMPLES_NUM_IN_DATA_RECORD(8),
        RESERVED(32);

        private final int id;

        EDFSignalParametersBytesNum(int id) {
            this.id = id;
        }

        public int getValue() {
            return id;
        }
    }

}