package multisab.processing.dataHandling;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 * @author Davor
 */

public class InputData {

    /**
     * The record name - the name of the file without extension
     */
    protected String recordName;
    /**
     * The record type - EDF, EDF+C, EDF+D, ANN, TXT
     */
    protected String recordType;
    protected String recordFilePath;
    protected Metadata metadata;
    /**
     * Number of ordinary signals
     */
    protected int numberOfSignals = 0;
    /**
     * Number of samples in the signal
     */
    protected int numberOfSamples[];
    /**
     * Duration of record (in seconds)
     */
    protected double duration;

    protected int numberOfAnnotations = 0;
    protected int numberOfAnnotsSamples[];

    protected double[][] signals;
    protected Annotation[][] annots;

    public InputData(File selectedFile) throws IOException {

        recordFilePath = selectedFile.getPath();
        String ext = "";

        int i = recordFilePath.lastIndexOf('.');
        if (i > 0) {
            ext = recordFilePath.substring(i + 1);
        }

        recordType = ext.toUpperCase();

        if (recordFilePath.contains("\\")) {
            this.recordName = recordFilePath.substring(recordFilePath.lastIndexOf("\\") + 1, recordFilePath.lastIndexOf("."));
        } else if (recordFilePath.contains("/")) {
            this.recordName = recordFilePath.substring(recordFilePath.lastIndexOf("/") + 1, recordFilePath.lastIndexOf("."));
        } else {
            this.recordName = recordFilePath.substring(0, recordFilePath.lastIndexOf("."));
        }
    }

    public static double timeStringToSeconds(String timeString) throws NumberFormatException {
        double temp = 0.0;
        int days = 0;
        if (timeString.charAt(0) == '[' && timeString.charAt(timeString.length() - 1) == ']') {
            timeString = timeString.substring(1, timeString.length() - 1);
            if (timeString.contains(" ")) {
                days = Integer.parseInt(timeString.substring(timeString.indexOf(" ") + 1));
            }
        }
        if (timeString.contains(":")) {
            if (timeString.indexOf(":") != timeString.lastIndexOf(":")) {
                temp += 3600 * Double.parseDouble(timeString.substring(0, timeString.indexOf(":")));
                timeString = timeString.substring(timeString.indexOf(":") + 1);
            }
            temp += 60 * Double.parseDouble(timeString.substring(0, timeString.indexOf(":")));
            if (!timeString.contains(" ")) {
                temp += Double.parseDouble(timeString.substring(timeString.indexOf(":") + 1));
            } else {
                temp += Double.parseDouble(timeString.substring(timeString.indexOf(":") + 1, timeString.indexOf(" ")));
            }
            temp += days * 3600 * 24;
        } else {
            temp = Double.parseDouble(timeString);
        }

        return temp;
    }

    /**
     * Pretvara vrijeme u sekundama u ljepši ispis u formatu sati:minute:sekunde.tisućinke.
     *
     * @param timeInSeconds Vrijeme zadano u sekundama, npr. 123.332
     * @return Formatizirani String u navedenom formatu.
     */
    public static String timeInSecondsToString(double timeInSeconds) {
        StringBuilder time = new StringBuilder();
        int hours = (int) timeInSeconds / 3600;
        int minutes = (int) timeInSeconds / 60;
        double timeInSeconds2;
        int timeInMinutes;
        if (hours == 0) {
            if (minutes == 0) {
                time.append(Double.toString(timeInSeconds));
            } else {
                timeInSeconds = timeInSeconds - minutes * 60;
                if (Double.toString(timeInSeconds).charAt(1) == '.') {
                    if (Double.toString(timeInSeconds).length() > 5) {
                        time.append(Integer.toString(minutes) + ":0" + Double.toString(timeInSeconds).substring(0, 5));
                    } else {
                        time.append(Integer.toString(minutes) + ":0" + Double.toString(timeInSeconds));
                    }
                } else {
                    if (Double.toString(timeInSeconds).length() > 6) {
                        time.append(Integer.toString(minutes) + ":" + Double.toString(timeInSeconds).substring(0, 6));
                    } else {
                        time.append(Integer.toString(minutes) + ":" + Double.toString(timeInSeconds));
                    }
                }
            }
        } else {
            timeInSeconds2 = timeInSeconds - hours * 3600;
            timeInMinutes = (int) (timeInSeconds2 / 60);
            timeInSeconds = timeInSeconds2 - timeInMinutes * 60;
            if (hours < 10) {
                time.append("0");
            }
            time.append(hours);
            time.append(":");
            if (timeInMinutes < 10) {
                time.append("0");
            }
            time.append(timeInMinutes);
            time.append(":");
            if (timeInSeconds < 10) {
                time.append("0");
            }
            if (Double.toString(timeInSeconds).length() > 5) {
                time.append(Double.toString(timeInSeconds).substring(0, 5));
            } else {
                time.append(Double.toString(timeInSeconds));
            }
        }
        return time.toString();
    }

    public String readBytesToString(BufferedInputStream input, int byteNum) throws IOException {
        byte[] byteInput = new byte[byteNum];
        int numByteRead = input.read(byteInput, 0, byteNum);
        if (numByteRead < byteNum) {
            System.out.println("An error occurred while loading file. The file must be closed.");
        }
        String dataString = new String(byteInput, "US-ASCII");
        return dataString;
    }

    public short[] convertBytesToShorts(byte[] bytes) {
        ByteBuffer bb = ByteBuffer.allocate(2);
        int k = 0;
        byte[] byteInput;
        short[] shorts = new short[bytes.length / 2];
        for (int i = 0; i < bytes.length - 1; i += 2) {
            byteInput = new byte[2];
            byteInput[0] = bytes[i];
            byteInput[1] = bytes[i + 1];
            bb.order(ByteOrder.LITTLE_ENDIAN);
            bb.put(byteInput);
            shorts[k] = bb.getShort(0);

            bb.clear();
            k++;
        }
        return shorts;
    }


    ///////////////////////////////////////////////////////////////////////////////////////
    ///                 Obavezne metode klase InputFile                                 ///
    ///////////////////////////////////////////////////////////////////////////////////////

    public long getSignalSamplesNum(int signalIndex) { // TODO promijeniti naziv
        return signals[signalIndex].length;
    }

    public long calculateSampleFromTime(double time, int signalIndex) {
        return Math.round(time * getSignalSamplesNum(signalIndex) / calculateDuration());
    }

    public double calculateTimeFromSample(double sample, int signalIndex) {
        return sample * calculateDuration() / getSignalSamplesNum(signalIndex);
    }

    public double calculateDuration() { // TODO: promijeniti naziv
        return duration; //metadata.getDataRecordsNum()* metadata.getDataRecordDurationInSec();
    }

    public double calculateFrequency(int signalIndex) {
        return (signals[signalIndex].length - 1) / duration;
    }

    public void findMinAndMaxSample(int signalIndex) {
        double maxSample, minSample, sample;
        maxSample = minSample = signals[signalIndex][0];

        for (int i = 0; i < numberOfSamples[signalIndex]; i++) {
            sample = signals[signalIndex][i];
            if (sample < minSample) {
                minSample = sample;
            } else if (sample > maxSample) {
                maxSample = sample;
            }
        }

        metadata.getSignalParameters()[signalIndex].setPhysicalSampleMax(maxSample);
        metadata.getSignalParameters()[signalIndex].setPhysicalSampleMin(minSample);
    }

    public void findMinAndMaxSampleAllSignals() {
        for (int i = 0; i < numberOfSignals; i++) { //TODO
            findMinAndMaxSample(i);
        }
    }

    public double[] getSamplesFromInterval(int signalIndex, long startSample, long endSample) {
        // TODO napravljeno za int
        double[] samplesFromInterval = new double[(int) (endSample - startSample + 1)];
        System.arraycopy(signals[signalIndex], (int) startSample, samplesFromInterval, 0, (int) (endSample - startSample + 1));

        return samplesFromInterval;
    }

    public static double[] getSamplesFromInterval(double[] signal, long startSample, long endSample) {
        // TODO napravljeno za int
        double[] samplesFromInterval = new double[(int) (endSample - startSample + 1)];
        System.arraycopy(signal, (int) startSample, samplesFromInterval, 0, (int) (endSample - startSample + 1));

        return samplesFromInterval;
    }

    public double[] getSamples(int signalIndex) {
        // TODO napravljeno za int
        double[] samples = new double[signals[signalIndex].length];
        System.arraycopy(signals[signalIndex], 0, samples, 0, signals[signalIndex].length);

        return samples;
    }

    public String getName() {
        return recordFilePath;
    }

    public Metadata getMetadata() {
        return metadata;
    }

    public void setMetadata(Metadata metadata) {
        this.metadata = metadata;
    }

    public Annotation[][] getAnnotations() {
        return annots;
    }
    
    public int getNumberOfSignals() {
        return numberOfSignals;
    }

    public int getNumberOfAnnotations() {
        return numberOfAnnotations;
    }

    public int[] getNumberOfAnnotsSamples() {
        return numberOfAnnotsSamples;
    }

    public Annotation[] getAnnotsFromInterval(int annotIndex, double startTime, double endTime) {
    	int numberOfAnnotsSamplesFromInterval = 0;
    	int i;
    	
    	for (i = 0; i < numberOfAnnotsSamples[annotIndex]; i++) {
            if ((annots[annotIndex][i].getTimeAnnots()) >= startTime && (annots[annotIndex][i].getTimeAnnots() <= endTime)) { //TODO za sada nije uključeno trajanje anotacije
            	numberOfAnnotsSamplesFromInterval = numberOfAnnotsSamplesFromInterval + 1;
            }
        }
    	
    	Annotation[] annotsForInterval = new Annotation[numberOfAnnotsSamplesFromInterval];
    	int j = 0;
    	
    	for (i = 0; i < numberOfAnnotsSamples[annotIndex]; i++) {
            if ((annots[annotIndex][i].getTimeAnnots()) >= startTime && (annots[annotIndex][i].getTimeAnnots() <= endTime)) {
            	annotsForInterval[j] = annots[annotIndex][i];
            	j = j + 1;
            }
        }
        
    	return annotsForInterval;
    }

    public double[][] getSignals() {
        return signals;
    }

    public double[] getSignal(int signalIndex){
        return signals[signalIndex];
    }

    public String toString() {
        return recordFilePath;
    }
    
}