package multisab.processing.analysis;

import multisab.processing.multisabException.ProcessingException;
import multisab.processing.scenarios.AnalysisType;
import multisab.processing.eegAnalysis.SpectralAnalysisEEG;
import multisab.processing.hrvAnalysis.SpectralAnalysisHRV;

import java.io.*;
import java.util.ArrayList;
import java.util.List;

public class PreprocessingMethod {
    public static final String FFT = "FFT";
    public static final String BURG = "BURG";
    public static final String LOMB_SCARGLE = "LOMB_SCARGLE";
    public static final String HILBERT_HUANG_TRANSFORM = "HILBERT_HUANG_TRANSFORM";
    public static final String PCA = "PCA";
    public static final String BASELINE_WANDER_REMOVAL_ECG = "BASELINE_WANDER_REMOVAL";
    public static final String MMF = "MMF";
    public static final String R_PEAKS_DETECTION_ECG = "R_PEAKS_DETECTION_ECG";
    public static final String ALL_AVAILABLE_PEAKS_DETECTION_ECG = "ALL_AVAILABLE_PEAKS_DETECTION_ECG";

    private String[] records;
    private String[][] signals;
    private boolean allRecords;
    private boolean allSignals;
    private String[] noiseFilteringMethods;
    private String[] morphologyMethods;
    private String[] transformationMethods;
    private String[] parametersForTransformationMethods;
    private String windowMethod;
    private String windowLength;
    private String start;
    private String end;
    private boolean untilEnd;
    private boolean parallelize;

    public PreprocessingMethod(String[] records,  String[][] signals,  boolean allRecords, boolean allSignals, String[] noiseFilteringMethods,  String[] morphologyMethods,  String[] transformationMethods, String[] parametersForTransformationMethods, String windowMethod,  String windowLength,  String start,  String end, boolean untilEnd, boolean parallelize){
        this.records = records;
        this.signals = signals;
        this.allRecords = allRecords;
        this.allSignals = allSignals;
        this.noiseFilteringMethods = noiseFilteringMethods;
        this.morphologyMethods = morphologyMethods;
        this.transformationMethods = transformationMethods;
        this.parametersForTransformationMethods= parametersForTransformationMethods;
        this.windowMethod = windowMethod;
        this.windowLength = windowLength;
        this.start = start;
        this.end = end;
        this.untilEnd = untilEnd;
        this.parallelize = parallelize;
    }

    public String[] getRecords(){
        return this.records;
    }
    public String [][] getSignals(){
        return this.signals;
    }
    public boolean getAllRecords(){
        return this.allRecords;
    }
    public boolean getAllSignals(){
        return this.allSignals;
    }
    public String[] getNoiseFilteringMethods(){
        return this.noiseFilteringMethods;
    }
    public String[] getMorphologyMethods(){
        return this.morphologyMethods;
    }
    public String[] getTransformationMethods(){
        return this.transformationMethods;
    }
    public String[] getParametersForTransformationMethods(){
        return this.parametersForTransformationMethods;
    }
    public String getWindowMethod(){
        return this.windowMethod;
    }
    public String getWindowLength(){
        return this.windowLength;
    }
    public String getStart(){
        return this.start;
    }
    public String getEnd(){
        return this.end;
    }
    public boolean getUntilEnd(){
        return this.untilEnd;
    }
    public boolean getParallelize(){
        return this.parallelize;
    }

    public static List<String> getAllPreprocessingMethodsForAnalysisAndDataType(String analysisType, String dataType) {
        List<String> list = new ArrayList<>();
        if (analysisType.equals(AnalysisType.INSPECTION)) {
            return list;
        } else if (analysisType.equals(AnalysisType.PREDICTION)) {
            //not implemented yet!
            return list;
        } else { //if classification or detection
            if (dataType.contains("ECG")) {
                list.add(PreprocessingMethod.R_PEAKS_DETECTION_ECG);
                list.add(PreprocessingMethod.ALL_AVAILABLE_PEAKS_DETECTION_ECG);
                list.add(PreprocessingMethod.BASELINE_WANDER_REMOVAL_ECG);
                list.add(PreprocessingMethod.MMF);
            }
            if (dataType.contains("ECG") || dataType.contains("EMG") || dataType.contains("EEG") || dataType.contains("OTHER")) {
                list.add(PreprocessingMethod.PCA);
                // add Hilbert Huang here once it is implemented and tested fully
            }
            if (dataType.contains("HRV")) {
                list.add(PreprocessingMethod.FFT);
                list.add(PreprocessingMethod.BURG);
                list.add(PreprocessingMethod.LOMB_SCARGLE);
            }
        }
        return list;
    }

    public static List<String> getAllSpecificPreprocessingMethods(String preprocessingMethod) {
        List<String> list = new ArrayList<>();

        switch (preprocessingMethod) {
            case "R_PEAKS_DETECTION_ECG":
                list.add("PAN_TOMPKINS_R_PEAK_DETECTION");
                list.add("ELGENDI_R_PEAK_DETECTION");
                break;
            case "ALL_AVAILABLE_PEAKS_DETECTION_ECG":
                list.add("PAN_TOMPKINS_R_PEAK_DETECTION");
                list.add("ELGENDI_R_PEAK_DETECTION");
                list.add("MARTINEZ_PT_FOR_P_AND_T_PEAKS");
                break;
        }
        return list;
    }


    public static boolean isNoiseFilteringMethod(String preprocessingMethod) {
        boolean isNoiseFiltering = false;

        switch (preprocessingMethod) {
            case PreprocessingMethod.BASELINE_WANDER_REMOVAL_ECG:
                isNoiseFiltering = true;
                break;
            case PreprocessingMethod.MMF:
                isNoiseFiltering = true;
                break;
        }

        return isNoiseFiltering;
    }


    public static boolean isMorphologyMethod(String preprocessingMethod) {
        boolean isMorphology = false;

        switch (preprocessingMethod) {
            case PreprocessingMethod.R_PEAKS_DETECTION_ECG:
                isMorphology = true;
                break;
            case PreprocessingMethod.ALL_AVAILABLE_PEAKS_DETECTION_ECG:
                isMorphology = true;
                break;
        }

        return isMorphology;
    }

    public static boolean isTransformation(String preprocessingMethod) {
        boolean isTransformation = false;

        switch (preprocessingMethod) {
            case PreprocessingMethod.BURG:
                isTransformation = true;
                break;
            case PreprocessingMethod.LOMB_SCARGLE:
                isTransformation = true;
                break;
            case PreprocessingMethod.FFT:
                isTransformation = true;
                break;
            case PreprocessingMethod.PCA:
                isTransformation = true;
                break;
        }

        return isTransformation;
    }

    public static void savePreprocessingMethodsToFile(String filename,  String[] records,  String[][] signals,  boolean allRecords, boolean allSignals, String[] noiseFilteringMethods,  String[] morphologyMethods,  String[] transformationMethods, String[] parametersForTransformationMethods, String windowMethod,  String windowLength,  String start,  String end, boolean untilEnd, boolean parallelize) throws ProcessingException {
        try {
            BufferedWriter bf;
            if (filename == null){
                bf = new BufferedWriter(new FileWriter("PreprocessingParameters.ana"));
            }
            else {
                bf = new BufferedWriter(new FileWriter(filename));
            }
            int i = 0, j = 0;
            bf.append("Records: ");
            if (allRecords){
                bf.append("ALL");
            }
            else {
                for (i = 0; i < records.length - 1; i++) {
                    bf.append(records[i] + ",");
                }
                if (records.length != 0) {
                    bf.append(records[i]);
                }
            }
            bf.newLine();
            bf.append("Signals: ");
            if (allSignals){
                bf.append("ALL");
                bf.newLine();
            }
            else {
                bf.newLine();
                for (i = 0; i < signals.length; i++) {
                    bf.append(records[i]+": ");
                    for (j = 0; j < signals[i].length-1; j++) {
                        bf.append(signals[i][j] + ",");
                    }
                    if (signals[i].length != 0) {
                        bf.append(signals[i][j]);
                    }
                    bf.newLine();
                }
            }
            bf.append("Noise filtering methods: ");
            for (i = 0; i < noiseFilteringMethods.length - 1; i++) {
                bf.append(noiseFilteringMethods[i] + ",");
            }
            if (noiseFilteringMethods.length != 0){
                bf.append(noiseFilteringMethods[i]);
            }
            bf.newLine();
            bf.append("Morphology methods: ");
            for (i = 0; i < morphologyMethods.length - 1; i++) {
                bf.append(morphologyMethods[i] + ",");
            }
            if (morphologyMethods.length != 0){
                bf.append(morphologyMethods[i]);
            }
            bf.newLine();
            bf.append("Transformation methods: ");
            for (i = 0; i < transformationMethods.length - 1; i++) {
                bf.append(transformationMethods[i] + ",");
            }
            if (transformationMethods.length != 0){
                bf.append(transformationMethods[i]);
            }
            bf.newLine();
            bf.append("Parameters for transformation methods: ");
            for (i = 0; i < parametersForTransformationMethods.length - 1; i++) {
                bf.append(parametersForTransformationMethods[i] + ",");
            }
            if (parametersForTransformationMethods.length != 0){
                bf.append(parametersForTransformationMethods[i]);
            }
            bf.newLine();
            bf.append("Window method: ");
            bf.append(windowMethod);
            bf.newLine();
            if (windowMethod.equals("detailed")){
                bf.append("Window length: ");
                bf.append(windowLength);
                bf.newLine();
                bf.append("Start: ");
                bf.append(start);
                bf.newLine();
                if (untilEnd){
                    bf.append("End: ");
                    bf.append("end of file");
                    bf.newLine();
                }
                else {
                    bf.append("End: ");
                    bf.append(end);
                    bf.newLine();
                }

            }
            bf.append("Parallelize: ");
            if (parallelize) {
                bf.append("yes");
            }
            else {
                bf.append("no");
            }
            bf.newLine();
            bf.flush();
            bf.close();
        }
        catch (IOException exc){
            throw new ProcessingException(exc.getMessage());
        }
        catch (Exception ex){
            throw new ProcessingException(ex.getMessage());
        }
    }

    public static String [] loadRecordsNamesFromPreprocessingFile(String filename) throws ProcessingException{
        String [] records = null;
        try {
            BufferedReader br;
            if (filename == null) {
                br = new BufferedReader(new FileReader("PreprocessingParameters.ana"));
            } else {
                br = new BufferedReader(new FileReader(filename));
            }
            int i = 0;
            String line = br.readLine();
            String temp;
            while(line != null){
                if (line.startsWith("Records: ")){
                    i = 0;
                    line = line.substring(line.indexOf(":")+2);
                    if (line.trim().equals("ALL")){
                        records = new String[1];
                        records[0] = "ALL";
                        break;
                    }
                    temp = line;
                    while (line.contains(",")){
                        line = line.substring(line.indexOf(",")+1);
                        i++;
                    }
                    if (line.trim().length() != 0) {
                        i++;
                    }
                    else {
                        break;
                    }
                    records = new String[i];
                    int j;
                    for (j = 0; j < i-1; j++){
                        records[j] = temp.substring(0,temp.indexOf(","));
                        temp = temp.substring(temp.indexOf(",")+1);
                    }
                    records[j] = temp;
                }
                line = br.readLine();
            }
            br.close();
            return records;
        }
        catch (IOException exc){
            throw new ProcessingException(exc.getMessage());
        }
        catch (Exception ex){
            throw new ProcessingException(ex.getMessage());
        }
    }

    public static List<String>[] loadSignalsNamesFromPreprocessingFile(String filename) throws ProcessingException{
        List<String>[] signalsList = null;
        try {
            BufferedReader br;
            if (filename == null) {
                br = new BufferedReader(new FileReader("PreprocessingParameters.ana"));
            } else {
                br = new BufferedReader(new FileReader(filename));
            }
            int i = 0;
            String line = br.readLine();
            while(line != null) {
                if (line.startsWith("Signals:")) {
                    if (line.substring(line.indexOf(":") + 1).trim().equals("ALL")) {
                        signalsList = new ArrayList[1];
                        signalsList[0] = new ArrayList<>();
                        signalsList[0].add("ALL");
                        br.close();
                        return signalsList;
                    }
                    i = 0;
                    line = br.readLine();
                    while (!line.startsWith("Noise filtering methods")) {
                        if (line != "" && line.contains(":")) {
                            i++;
                        }
                        line = br.readLine();
                    }
                    signalsList = new ArrayList[i];
                    break;
                }
                line = br.readLine();
            }
            br.close();
            if (filename == null) {
                br = new BufferedReader(new FileReader("PreprocessingParameters.ana"));
            }
            else {
                br = new BufferedReader(new FileReader(filename));
            }
            line = br.readLine();
            /*
            Signals:
            chf01.ann: ANN_Annotations
            chf02.ann: ANN_Annotations
             */
            //
            while(!line.startsWith("Noise filtering methods") && line != null) {
                if (line.startsWith("Signals:")) {
                    int j;
                    line = br.readLine();
                    for (i = 0; i <signalsList.length; i++) {
                        if (line == "" || !line.contains(":")) {
                            i--;
                            line = br.readLine();
                        }
                        else {
                            line = line.substring(line.indexOf(":")+1);
                            signalsList[i] = new ArrayList<>();
                            while (line.contains(",")){
                                signalsList[i].add(line.substring(0,line.indexOf(",")).trim());
                                line = line.substring(line.indexOf(",")+1);
                            }
                            signalsList[i].add(line.trim());
                        }
                    }
                }
                line = br.readLine();
            }
            br.close();
            return signalsList;
        }
        catch (IOException exc){
            throw new ProcessingException(exc.getMessage());
        }
        catch (Exception ex){
            throw new ProcessingException(ex.getMessage());
        }
    }

    public static String [] loadNoiseFilteringMethodsFromPreprocessingFile(String filename) throws ProcessingException{
        String [] noiseFilteringMethods = null;
        try {
            BufferedReader br;
            if (filename == null) {
                br = new BufferedReader(new FileReader("PreprocessingParameters.ana"));
            } else {
                br = new BufferedReader(new FileReader(filename));
            }
            int i = 0;
            String line = br.readLine();
            String temp;
            while(line != null){
                if (line.startsWith("Noise filtering methods:")){
                    i = 0;
                    line = line.substring(line.indexOf(":")+2);
                    temp = line;
                    while (line.contains(",")){
                        line = line.substring(line.indexOf(",")+1);
                        i++;
                    }
                    if (line.trim().length() != 0) {
                        i++;
                    }
                    else {
                        break;
                    }
                    noiseFilteringMethods = new String[i];
                    int j;
                    for (j = 0; j < i-1; j++){
                        noiseFilteringMethods[j] = temp.substring(0,temp.indexOf(","));
                        temp = temp.substring(temp.indexOf(",")+1);
                    }
                    noiseFilteringMethods[j] = temp;
                }
                line = br.readLine();
            }
            br.close();
            return noiseFilteringMethods;
        }
        catch (IOException exc){
            throw new ProcessingException(exc.getMessage());
        }
        catch (Exception ex){
            throw new ProcessingException(ex.getMessage());
        }
    }

    public static String [] loadMorphologyMethodsFromPreprocessingFile(String filename) throws ProcessingException{
        String [] morphologyMethods = null;
        try {
            BufferedReader br;
            if (filename == null) {
                br = new BufferedReader(new FileReader("PreprocessingParameters.ana"));
            } else {
                br = new BufferedReader(new FileReader(filename));
            }
            int i;
            String line = br.readLine();
            String temp;
            while(line != null){
                if (line.startsWith("Morphology methods:")){
                    i = 0;
                    line = line.substring(line.indexOf(":")+2);
                    temp = line;
                    while (line.contains(",")){
                        line = line.substring(line.indexOf(",")+1);
                        i++;
                    }
                    if (line.trim().length() != 0) {
                        i++;
                    }
                    else {
                        break;
                    }
                    morphologyMethods = new String[i];
                    int j;
                    for (j = 0; j < i-1; j++){
                        morphologyMethods[j] = temp.substring(0,temp.indexOf(","));
                        temp = temp.substring(temp.indexOf(",")+1);
                    }
                    morphologyMethods[j] = temp;
                }
                line = br.readLine();
            }
            br.close();
            return morphologyMethods;
        }
        catch (IOException exc){
            throw new ProcessingException(exc.getMessage());
        }
        catch (Exception ex){
            throw new ProcessingException(ex.getMessage());
        }
    }

    public static String [] loadTransformationMethodsFromPreprocessingFile(String filename) throws ProcessingException{
        String [] transformationMethods = null;
        try {
            BufferedReader br;
            if (filename == null) {
                br = new BufferedReader(new FileReader("PreprocessingParameters.ana"));
            } else {
                br = new BufferedReader(new FileReader(filename));
            }
            int i;
            String line = br.readLine();
            String temp;
            while(line != null){
                if (line.startsWith("Transformation methods:")){
                    i = 0;
                    line = line.substring(line.indexOf(":")+2);
                    temp = line;
                    while (line.contains(",")){
                        line = line.substring(line.indexOf(",")+1);
                        i++;
                    }
                    if (line.trim().length() != 0) {
                        i++;
                    }
                    else {
                        break;
                    }
                    transformationMethods = new String[i];
                    int j;
                    for (j = 0; j < i-1; j++){
                        transformationMethods[j] = temp.substring(0,temp.indexOf(","));
                        temp = temp.substring(temp.indexOf(",")+1);
                    }
                    transformationMethods[j] = temp;
                }
                line = br.readLine();
            }
            br.close();
            return transformationMethods;
        }
        catch (IOException exc){
            throw new ProcessingException(exc.getMessage());
        }
        catch (Exception ex){
            throw new ProcessingException(ex.getMessage());
        }
    }

    public static String [] loadParametersForTransformationMethodsFromPreprocessingFile(String filename) throws ProcessingException{
        String [] parametersForTransformationMethods = null;
        try {
            BufferedReader br;
            if (filename == null) {
                br = new BufferedReader(new FileReader("PreprocessingParameters.ana"));
            } else {
                br = new BufferedReader(new FileReader(filename));
            }
            int i;
            String line = br.readLine();
            String temp;
            while(line != null){
                if (line.startsWith("Parameters for transformation methods:")){
                    i = 0;
                    line = line.substring(line.indexOf(":")+2);
                    temp = line;
                    while (line.contains(",")){
                        line = line.substring(line.indexOf(",")+1);
                        i++;
                    }
                    if (line.trim().length() != 0) {
                        i++;
                    }
                    else {
                        break;
                    }
                    parametersForTransformationMethods = new String[i];
                    int j;
                    for (j = 0; j < i-1; j++){
                        parametersForTransformationMethods[j] = temp.substring(0,temp.indexOf(","));
                        temp = temp.substring(temp.indexOf(",")+1);
                    }
                    parametersForTransformationMethods[j] = temp;
                }
                line = br.readLine();
            }
            br.close();
            return parametersForTransformationMethods;
        }
        catch (IOException exc){
            throw new ProcessingException(exc.getMessage());
        }
        catch (Exception ex){
            throw new ProcessingException(ex.getMessage());
        }
    }

    public static String [] loadWindowingMethodFromPreprocessingFile(String filename) throws ProcessingException{
        String [] windowingMethod = null;
        try {
            BufferedReader br;
            if (filename == null) {
                br = new BufferedReader(new FileReader("PreprocessingParameters.ana"));
            } else {
                br = new BufferedReader(new FileReader(filename));
            }
            String line = br.readLine();
            while(line != null){
                if (line.startsWith("Window method:")){
                    line = line.substring(line.indexOf(":")+2);
                    if (line.trim().equals("detailed")){
                        windowingMethod = new String[4];
                        windowingMethod[0] = line.trim();
                        line = br.readLine();
                        windowingMethod[1] = line.substring(line.indexOf(":")+1).trim(); // window length (in seconds)
                        line = br.readLine();
                        windowingMethod[2] = line.substring(line.indexOf(":")+1).trim(); // start (in seconds)
                        line = br.readLine();
                        windowingMethod[3] = line.substring(line.indexOf(":")+1).trim(); // end (in seconds, or "end of file")
                    }
                    else {
                        windowingMethod = new String[1];
                        windowingMethod[0] = line.trim();
                    }
                    break;
                }
                line = br.readLine();
            }
            br.close();
            return windowingMethod;
        }
        catch (IOException exc){
            throw new ProcessingException(exc.getMessage());
        }
        catch (Exception ex){
            throw new ProcessingException(ex.getMessage());
        }
    }

    public static boolean loadParallelizePreprocessingFromPreprocessingFile(String filename) throws ProcessingException{
        String [] windowingMethod = null;
        try {
            BufferedReader br;
            if (filename == null) {
                br = new BufferedReader(new FileReader("PreprocessingParameters.ana"));
            } else {
                br = new BufferedReader(new FileReader(filename));
            }
            String line = br.readLine();
            while(line != null){
                if (line.startsWith("Parallelize:")){
                    line = line.substring(line.indexOf(":")+2);
                    if (line.trim().equals("yes")){
                        br.close();
                        return true;
                    }
                    else {
                        br.close();
                        return false;
                    }
                }
                line = br.readLine();
            }
            br.close();
            return false;
        }
        catch (IOException exc){
            throw new ProcessingException(exc.getMessage());
        }
        catch (Exception ex){
            throw new ProcessingException(ex.getMessage());
        }
    }

    public static String getParametersForPreprocessingMethod(String method){
        String parameters = "";

        switch (method){
            case PreprocessingMethod.FFT:
                parameters = "window="+SpectralAnalysisEEG.HAMMING_WINDOW+";";
                break;
            case PreprocessingMethod.BURG:
                parameters = "ord=12;";
                break;
            default:
                parameters = "";
                break;

        }
        return parameters;
    }

}
