package multisab.processing.eegAnalysis;

import multisab.processing.commonSignalFeatures.frequencyDomain.SpectralAnalysis;
import multisab.processing.multisabException.ProcessingException;

public class SpectralAnalysisEEG {
    public static final double DEFAULT_ALPHA_BAND_LOWER_BOUND = 7.5;
    public static final double DEFAULT_ALPHA_BAND_UPPER_BOUND = 12.5;
    public static final double DEFAULT_BETA_BAND_LOWER_BOUND = 13;
    public static final double DEFAULT_BETA_BAND_UPPER_BOUND = 25;
    public static final double DEFAULT_GAMMA_BAND_LOWER_BOUND = 25;
    public static final double DEFAULT_GAMMA_BAND_UPPER_BOUND = 100;
    public static final double DEFAULT_DELTA_BAND_LOWER_BOUND = 1;
    public static final double DEFAULT_DELTA_BAND_UPPER_BOUND = 4;
    public static final double DEFAULT_THETA_BAND_LOWER_BOUND = 4;
    public static final double DEFAULT_THETA_BAND_UPPER_BOUND = 7;
    public static final int FAST_FOURIER_TRANSFORM = 1;
    public static final int BURG_METHOD = 2;
    public static final int DEFAULT_ORDER = 12;

	/*
 	* Spectral entropy estimation based on:
    * Ermes, M., Pärkkä, J. & Cluitmans, L. 2008. Advancing from Offline to
    * Online Activity Recognition with Wearable Sensors. Proceedings of the
	* 30th Annual International Conference of the IEEE Engineering in Medicine
	* and Biology Society. Pp. 4451–4454. 
	* 
	*  A. Rezek and S. J. Roberts, “Stochastic Complexity Measures for Physiological Signal Analysis,” IEEE Trans. Biomed. Eng., vol. 45, no. 9, pp. 1186–1191, Sep. 1998.
	*
	* The original paper uses fast Fourier transform to obtain spectrum and calculate the entropy.
	* This method may be used to calculate Spectral entropy using any previously obtained PSD estimate
    * @return spectral entropy
	*/
    public static final int LOMB_SCARGLE_METHOD = 3;
    public static final int NO_WINDOW = 0;
    public static final int HAMMING_WINDOW = 1;
    public static final int HANN_WINDOW = 2;
    SpectralAnalysis spectEEG = null;
    double[] frequencies;
    double[] psdEstimate;
    public SpectralAnalysisEEG(double[] segment, double sampFreq, int method, String window, int order) throws ProcessingException {
        spectEEG = new SpectralAnalysis(segment, sampFreq);
        boolean success = false;
        switch (method) {
            case SpectralAnalysis.FAST_FOURIER_TRANSFORM:
                success = spectEEG.calculateSpectrumFourier(window);
                break;
            case SpectralAnalysis.BURG_METHOD:
                success = spectEEG.calculateSpectrumBurg(order);
                break;
            case SpectralAnalysis.LOMB_SCARGLE_METHOD:
                success = spectEEG.calculateSpectrumLombScargle();
                break;
        }
        if (!success) throw new ProcessingException("Unable to calculate EEG spectrum.");
        frequencies = spectEEG.getFrequencies();
        psdEstimate = spectEEG.getPSDEstimate();
    }

    public double getTotalPSD() {
        double sum = 0.0;
        for (int i = 0; i < this.frequencies.length; i++) {
            sum += psdEstimate[i];
        }
        return sum;
    }

    /**
     * A method for getting PSD within a defined frequency band
     *
     * @param lowerFrequency Lower bound of the band
     * @param upperFrequency Upper bound of the band
     * @return Power spectral density estimate
     */
    public double getPSDForFrequencyBand(double lowerFrequency, double upperFrequency) {
        double sum = 0.0;
        int minIndex = 0;
        int maxIndex = 0;
        int i;
        for (i = 0; i < this.frequencies.length; i++) {
            if (frequencies[i] >= lowerFrequency) {
                break;
            }
        }
        minIndex = i;
        for (i = minIndex; i < this.frequencies.length; i++) {
            if (frequencies[i] > upperFrequency) {
                break;
            }
        }
        maxIndex = i - 1;

        for (i = minIndex; i <= maxIndex; i++) {
            sum += psdEstimate[i];
        }
        return sum;
    }

    public double getFrequencyBand(double lowerFrequency, double upperFrequency) {
        double sum = 0.0;
        int minIndex = 0;
        int maxIndex = 0;
        int i;
        for (i = 0; i < this.frequencies.length; i++) {
            if (frequencies[i] >= lowerFrequency) {
                break;
            }
        }
        minIndex = i;
        for (i = minIndex; i < this.frequencies.length; i++) {
            if (frequencies[i] > upperFrequency) {
                break;
            }
        }
        maxIndex = i;

        double[] bandFrequencies = new double[maxIndex - minIndex];
        for (i = minIndex; i < maxIndex; i++) {
            bandFrequencies[i] = this.frequencies[i];
        }
        return sum;
    }

    public double getAlpha() {
        return this.getPSDForFrequencyBand(DEFAULT_ALPHA_BAND_LOWER_BOUND, DEFAULT_ALPHA_BAND_UPPER_BOUND);
    }

    public double getBeta() {
        return this.getPSDForFrequencyBand(DEFAULT_BETA_BAND_LOWER_BOUND, DEFAULT_BETA_BAND_UPPER_BOUND);
    }

    public double getGamma() {
        return this.getPSDForFrequencyBand(DEFAULT_GAMMA_BAND_LOWER_BOUND, DEFAULT_GAMMA_BAND_UPPER_BOUND);
    }

    public double getDelta() {
        return this.getPSDForFrequencyBand(DEFAULT_DELTA_BAND_LOWER_BOUND, DEFAULT_DELTA_BAND_UPPER_BOUND);
    }

    public double getTheta() {
        return this.getPSDForFrequencyBand(DEFAULT_THETA_BAND_LOWER_BOUND, DEFAULT_THETA_BAND_UPPER_BOUND);
    }

    /**
     * This method may be used to calculate Spectral entropy using any previously obtained PSD estimate
     *
     * @return
     */
    public double getSpectralEntropy() {
        double spectEntropy = 0;
        int i;

        for (i = 0; i < frequencies.length; i++) {
            spectEntropy += psdEstimate[i] * Math.log10(frequencies[i]);
        }
        spectEntropy = -spectEntropy / Math.log10((double) (frequencies.length));

        return spectEntropy;
    }
}
