package multisab.processing.preprocessing.filtering;

/**
 * Class FilterTransferFunctionBuilder is used to compute denominator and numerator coefficients of digital filter
 * represented in transfer function in Z domain:
 * H(z) = (b0 + b1 * z^-1 + b2 * z^-2 + ... + bN * z^-N)/ (1 + a1 * z^-1 + a2 * z^-2 + ... + aM * z^-M)
 *
 * Current implementation supports:
 * 1) Low and High Pass Biquad filters
 * 2) Lowpass Butterworth filter of order 2
 *
 * When creating new filter:
 * @input f0 - cutting frequency
 * @input fs - sampling frequency
 * @input type - 'lowBiquad', 'highBiquad', 'lowButter'
 *
 * @output b nominator
 * @output a denominator
 *
 *
 *
 * References:
 * Biquad filters: http://www.musicdsp.org/files/Audio-EQ-Cookbook.txt
 *
 *
 *
 * @author: Krešimir Friganović
 * @date: 9.12.2016.
 */
public class FilterTransferFunction {

    private static int n  = 2;
    private static double[] b = new double[n+1];
    private static double[] a = new double[n+1];


    public FilterTransferFunction(double f0, double fs, String type) {
        if (type == "lowBiquad") {
            double w0;
            double alpha;
            double Q = 1;
            w0 = (2 * Math.PI * f0) / fs;
            alpha = (Math.sin(w0)) / (2 * Q);

            b[0] = (1 - Math.cos(w0)) / 2;
            b[1] = 2 * b[0];
            b[2] = b[0];

            a[0] = 1 + alpha;
            a[1] = -2 * Math.cos(w0);
            a[2] = 1 - alpha;
        }

        if (type == "highBiquad") {
            double w0;
            double alpha;
            double Q = 0.1;
            w0 = (2 * Math.PI * f0) / fs;
            alpha = (Math.sin(w0)) / (2 * Q);

            b[0] = (1 + Math.cos(w0)) / 2;
            b[1] = -1 - Math.cos(w0);
            b[2] = b[0];

            a[0] = 1 + alpha;
            a[1] = -2 * Math.cos(w0);
            a[2] = 1 - alpha;
        }

        if (type == "lowButter") {
            double wc = Math.tan(Math.PI * f0 / fs);
            double C = 1 + 2 * Math.cos(Math.PI / 4) * wc + Math.pow(wc, 2);

            b[0] = Math.pow(wc, 2) / C;
            b[1] = (double) 2 * b[0];
            b[2] = b[0];

            a[0] = 1;
            a[1] = 2 * (Math.pow(wc, 2) - 1) / C;
            a[2] = (1 - 2 * Math.cos(Math.PI / 4) * wc + Math.pow(wc, 2)) / C;

        }
    }
    //Getters for b and a
    public static double[] getA() { return a;}
    public static double[] getB() { return b;}

}









