package multisab.processing.analysis;

import multisab.processing.commonSignalFeatures.nonlinear.entropy.AlphabetEntropy;
import multisab.processing.multisabException.FeatureReadingException;

import java.io.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

/**
 * @author Alan Jović
 */
public class ExtractedFeatures {
    private static final String ORD_NUMBER = "Ordinal_number";
    private static final String FILE_LABEL = "Input_file_label";
    private static final String SIGNAL_LABEL = "Signal_label";
    private static final String START_TIME_SEC = "Start_time_seconds";
    private static final String SEGMENT_LENGTH_SEC = "Segment_length_seconds";
    private static final String DISORDER = "Disorder";
    private ArrayList<String> ordNumber;
    private ArrayList<String> filename;
    private ArrayList<String> signalLabel;
    private ArrayList<String> startTime;
    private ArrayList<String> segmentLength;
    private ArrayList<String> featureName;
    private ArrayList<String[]> extractedFeatures;
    private ArrayList<String> disorder;

    public ExtractedFeatures(String[] featureName, HashMap<String,String> featureParameters, boolean addDisorderAttribute) {
        ordNumber = new ArrayList<>();
        filename = new ArrayList<>();
        startTime = new ArrayList<>();
        segmentLength = new ArrayList<>();
        signalLabel = new ArrayList<>();
        extractedFeatures = new ArrayList<String[]>();
        if (addDisorderAttribute) {
            disorder = new ArrayList<>();
        }
        this.featureName = new ArrayList<String>();
        this.featureName.add(ExtractedFeatures.ORD_NUMBER);
        this.featureName.add(ExtractedFeatures.FILE_LABEL);
        this.featureName.add(ExtractedFeatures.SIGNAL_LABEL);
        this.featureName.add(ExtractedFeatures.START_TIME_SEC);
        this.featureName.add(ExtractedFeatures.SEGMENT_LENGTH_SEC);

        String parameter;
        ArrayList<Integer> m;
        ArrayList<Double> r;

        // this implementation is feature-specific, it should be modified if some features contain special parameters, and commented for that features
        for (int i = 0; i < featureName.length; i++) {
            if (featureParameters.containsKey(featureName[i])){
                switch (featureName[i]){
                    case Features.AP_EN:        // approximate entropy has two parameters: m and r, there can be multiple m and r. multiple r are always considered for some m
                        parameter = featureParameters.get(Features.AP_EN);
                        m = new ArrayList<Integer>();
                        while(parameter.contains("m=")){
                            m.add(Integer.parseInt(parameter.substring(parameter.indexOf("m=")+2,parameter.indexOf(";", parameter.indexOf("m=")))));
                            parameter = parameter.substring(0,parameter.indexOf("m="))+parameter.substring(parameter.indexOf(";", parameter.indexOf("m=")));
                        }
                        r = new ArrayList<Double>();
                        while(parameter.contains("r=")) {
                            r.add(Double.parseDouble(parameter.substring(parameter.indexOf("r=")+2, parameter.indexOf(";", parameter.indexOf("r=")))));
                            parameter = parameter.substring(0, parameter.indexOf("r=")) + parameter.substring(parameter.indexOf(";", parameter.indexOf("r=")));
                        }
                        for (int j=0; j<m.size(); j++){
                            for (int k=0; k<r.size(); k++){
                                this.featureName.add(Features.AP_EN + "_m="+m.get(j)+"_r="+r.get(k));
                            }
                        }
                        break;
                    case Features.SAMP_EN:              // sample entropy has two parameters: m and r, there can be multiple m and r. multiple r are always considered for some m
                        parameter = featureParameters.get(Features.SAMP_EN);
                        m = new ArrayList<Integer>();
                        while(parameter.contains("m=")){
                            m.add(Integer.parseInt(parameter.substring(parameter.indexOf("m=")+2,parameter.indexOf(";", parameter.indexOf("m=")))));
                            parameter = parameter.substring(0,parameter.indexOf("m="))+parameter.substring(parameter.indexOf(";", parameter.indexOf("m=")));
                        }
                        r = new ArrayList<Double>();
                        while(parameter.contains("r=")) {
                            r.add(Double.parseDouble(parameter.substring(parameter.indexOf("r=")+2, parameter.indexOf(";", parameter.indexOf("r=")))));
                            parameter = parameter.substring(0, parameter.indexOf("r=")) + parameter.substring(parameter.indexOf(";", parameter.indexOf("r=")));
                        }
                        for (int j=0; j<m.size(); j++){
                            for (int k=0; k<r.size(); k++){
                                this.featureName.add(Features.SAMP_EN + "_m="+m.get(j)+"_r="+r.get(k));
                            }
                        }
                        break;
                    case Features.ALPHABET_ENTROPY:     // alphabet entropy contains many features:
                        // letter exists, letter rate, average alphabet entropy, alphabet entropy variance, maximum alphabet entropy,
                        // average alphabet entropy per letter, variance of alphabet entropy per letter, maximum alphabet entropy per letter
                        // here it is assumed that these features are not parametric, but that all of them are extracted
                        for (int j = 0; j<27; j++){
                            this.featureName.add(AlphabetEntropy.letterToString(j)+"_exists");
                        }
                        for (int j = 0; j<27; j++){
                            this.featureName.add(AlphabetEntropy.letterToString(j)+"_rate");
                        }
                        this.featureName.add("AverageAlphabetEntropy");
                        this.featureName.add("AlphabetEntropyVariance");
                        this.featureName.add("MaximumAlphabetEntropy");
                        for (int j = 0; j<27; j++){
                            this.featureName.add(AlphabetEntropy.letterToString(j)+"_average_entropy");
                        }
                        for (int j = 0; j<27; j++){
                            this.featureName.add(AlphabetEntropy.letterToString(j)+"_entropy_variance");
                        }
                        for (int j = 0; j<27; j++){
                            this.featureName.add(AlphabetEntropy.letterToString(j)+"_maximum_entropy");
                        }
                        break;
                    case Features.FUZZY_AP_EN:              // sample entropy has two parameters: m and r, there can be multiple m and r. multiple r are always considered for some m
                        parameter = featureParameters.get(Features.FUZZY_AP_EN);
                        m = new ArrayList<Integer>();
                        while(parameter.contains("m=")){
                            m.add(Integer.parseInt(parameter.substring(parameter.indexOf("m=")+2,parameter.indexOf(";", parameter.indexOf("m=")))));
                            parameter = parameter.substring(0,parameter.indexOf("m="))+parameter.substring(parameter.indexOf(";", parameter.indexOf("m=")));
                        }
                        r = new ArrayList<Double>();
                        while(parameter.contains("r=")) {
                            r.add(Double.parseDouble(parameter.substring(parameter.indexOf("r=")+2, parameter.indexOf(";", parameter.indexOf("r=")))));
                            parameter = parameter.substring(0, parameter.indexOf("r=")) + parameter.substring(parameter.indexOf(";", parameter.indexOf("r=")));
                        }
                        for (int j=0; j<m.size(); j++){
                            for (int k=0; k<r.size(); k++){
                                this.featureName.add(Features.FUZZY_AP_EN + "_m="+m.get(j)+"_r="+r.get(k));
                            }
                        }
                        break;
                    case Features.RECURRENCE_PLOT:
                        parameter = featureParameters.get(Features.RECURRENCE_PLOT);
                        String temp = "";
                        if (parameter.contains(";")){
                            while (parameter.contains(";")){
                                temp += "_"+parameter.substring(0,parameter.indexOf(";"));
                                parameter = parameter.substring(parameter.indexOf(";")+1);
                            }
                            this.featureName.add(Features.RECURRENCE_PLOT+temp+"_RecurrenceRate");
                            this.featureName.add(Features.RECURRENCE_PLOT+temp+"_LMean");
                            this.featureName.add(Features.RECURRENCE_PLOT+temp+"_DET");
                            this.featureName.add(Features.RECURRENCE_PLOT+temp+"_ShannonEntropyRecurrence");
                            this.featureName.add(Features.RECURRENCE_PLOT+temp+"_Laminarity");
                        }
                        break;
                        // add some more special cases if applicable
                    default:    // this is the default implementation for all features that contain regular parameters
                        parameter = featureParameters.get(featureName[i]);
                        if (parameter.contains(";")){
                            while (parameter.contains(";")){
                                featureName[i] += "_"+parameter.substring(0,parameter.indexOf(";"));
                                parameter = parameter.substring(parameter.indexOf(";")+1);
                            }
                            this.featureName.add(featureName[i]);
                        }
                        break;
                }
            }
            else {  // a feature that has no parameters is simply added to the list of features
                this.featureName.add(featureName[i]);
            }
        }
        if (addDisorderAttribute) {
            this.featureName.add(ExtractedFeatures.DISORDER);
        }
    }

    public void addExtractedFeatureVector(String filename, String signalLabel, String startTime, String segmentLength, String[] extractedFeatures, String disorder) {
        this.ordNumber.add(""+ordNumber.size());
        this.filename.add("\""+filename+"\"");
        this.signalLabel.add("\""+signalLabel+"\"");
        this.startTime.add(startTime);
        this.segmentLength.add(segmentLength);
        this.extractedFeatures.add(extractedFeatures);
        if (disorder != null) {
            this.disorder.add(disorder);
        }
    }

    public String[] getAllFeatureNames() {
        Iterator<String> it = this.featureName.iterator();
        String [] featureNames = new String[featureName.size()];
        int i=0;
        while (it.hasNext()){
            featureNames[i] = it.next();
            i++;
        }
        return featureNames;
    }

    public int getExtractedFeatureVectorsCount() {
        return this.extractedFeatures.size();
    }

    public String getOrdNumber(int i) {
        return this.ordNumber.get(i);
    }

    public String getFilename(int i) {
        return this.filename.get(i);
    }

    public String getSignalLabel(int i) {
        return this.signalLabel.get(i);
    }

    public String getStartTime(int i) {
        return this.startTime.get(i);
    }

    public String getSegmentLength(int i) {
        return this.segmentLength.get(i);
    }

    public String[] getExtractedFeatures(int i) {
        return this.extractedFeatures.get(i);
    }

    public String getDisorder(int i) {
        if (this.disorder != null) {
            return this.disorder.get(i);
        }
        else return null;
    }

    public void writeFeatureExtractionResultsToOutputFile(String filename){
        String reducedFileName = filename.substring(0,filename.lastIndexOf("."));
        try {
            BufferedWriter bf = new BufferedWriter(new FileWriter(reducedFileName+".csv"));
            int i,j;
            for (i = 0; i< this.getAllFeatureNames().length-1; i++){
                bf.append(this.getAllFeatureNames()[i]+",");
            }
            bf.append(this.getAllFeatureNames()[i]);
            bf.newLine();
            for (i = 0; i< this.getExtractedFeatureVectorsCount(); i++) {
                bf.append(this.getOrdNumber(i)+","+this.getFilename(i)+","+this.getSignalLabel(i)+","+this.getStartTime(i)+","+this.getSegmentLength(i)+",");
                for (j = 0; j< this.getExtractedFeatures(i).length-1; j++){
                    if (this.getExtractedFeatures(i)[j].substring(this.getExtractedFeatures(i)[j].indexOf(".")+1).length() > Parallelization.DECIMAL_PLACES){
                        bf.append(this.getExtractedFeatures(i)[j].substring(0,this.getExtractedFeatures(i)[j].indexOf(".")+1 + Parallelization.DECIMAL_PLACES)+",");
                    }
                    else {
                        bf.append(this.getExtractedFeatures(i)[j]+",");
                    }
                }
                if (this.getExtractedFeatures(i)[j].substring(this.getExtractedFeatures(i)[j].indexOf(".")+1).length() > Parallelization.DECIMAL_PLACES){
                    bf.append(this.getExtractedFeatures(i)[j].substring(0,this.getExtractedFeatures(i)[j].indexOf(".")+1 + Parallelization.DECIMAL_PLACES));
                }
                else {
                    bf.append(this.getExtractedFeatures(i)[j]);
                }
                if (this.getDisorder(i) != null) {
                    bf.append(","+this.getDisorder(i));
                }
                bf.newLine();
            }
            bf.flush();
            bf.close();
        }
        catch (IOException exc){
            exc.printStackTrace();
        }
        try {
            BufferedWriter bf = new BufferedWriter(new FileWriter(reducedFileName+".arff"));
            int i = 0,j;
            bf.append("@relation biomedical_signal_analysis");
            bf.newLine();
            bf.newLine();
            if (disorder != null) {
                for (i = 0; i < this.getAllFeatureNames().length - 1; i++) {
                    if (i == 0 || i >= 3) {
                        bf.append("@attribute " + this.getAllFeatureNames()[i] + " numeric");
                    } else {
                        bf.append("@attribute " + this.getAllFeatureNames()[i] + " string");
                    }
                    bf.newLine();
                }
                String[] disorders = Disorders.getAllDisorders();
                bf.append("@attribute " + this.getAllFeatureNames()[i]);
                if (disorders != null){
                    if (disorders.length != 0){
                        bf.append("{");
                        for (i = 0; i < disorders.length-1; i++){
                            bf.append(disorders[i]+",");
                        }
                        bf.append(disorders[i]+"}");
                    }
                }
            }
            bf.newLine();
            bf.newLine();
            bf.append("@data");
            bf.newLine();
            for (i = 0; i< this.getExtractedFeatureVectorsCount(); i++) {
                bf.append(this.getOrdNumber(i)+","+this.getFilename(i)+","+this.getSignalLabel(i)+","+this.getStartTime(i)+","+this.getSegmentLength(i)+",");
                for (j = 0; j< this.getExtractedFeatures(i).length-1; j++){
                    if (this.getExtractedFeatures(i)[j].substring(this.getExtractedFeatures(i)[j].indexOf(".")+1).length() > Parallelization.DECIMAL_PLACES){
                        bf.append(this.getExtractedFeatures(i)[j].substring(0,this.getExtractedFeatures(i)[j].indexOf(".")+1 + Parallelization.DECIMAL_PLACES)+",");
                    }
                    else {
                        bf.append(this.getExtractedFeatures(i)[j]+",");
                    }
                }
                if (this.getExtractedFeatures(i)[j].substring(this.getExtractedFeatures(i)[j].indexOf(".")+1).length() > Parallelization.DECIMAL_PLACES){
                    bf.append(this.getExtractedFeatures(i)[j].substring(0,this.getExtractedFeatures(i)[j].indexOf(".")+1 + Parallelization.DECIMAL_PLACES));
                }
                else {
                    bf.append(this.getExtractedFeatures(i)[j]);
                }
                if (this.getDisorder(i) != null) {
                    bf.append(","+this.getDisorder(i));
                }
                bf.newLine();
            }
            bf.flush();
            bf.close();
        }
        catch (IOException exc){
            exc.printStackTrace();
        }
    }

    public static String[][] readAllExtractedFeaturesAsString(String fileName) throws FeatureReadingException{
        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            }
            else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == ""){
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                }
                else {
                    throw new FeatureReadingException("No features extracted in file "+fileName);
                }
            }
            // the first row should contain features' names, use it to discover the number of features
            int noOfFeatures = 0;
            while (s.contains(",")){
                s = s.substring(s.indexOf(",")+1);
                noOfFeatures++;
            }
            noOfFeatures++;
            int countFeatureVectors = 0;
            s = br.readLine();
            while (s != null){
                countFeatureVectors++;
                s = br.readLine();
            }
            br.close();

            if (countFeatureVectors == 0 || noOfFeatures <= 5){
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                }
                else {
                    throw new FeatureReadingException("No features extracted in file "+fileName);
                }
            }
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            }
            else {
                br = new BufferedReader(new FileReader(fileName));
            }
            s = br.readLine(); // the first row should contain features' names, ignore it now
            String [][] features = new String[countFeatureVectors][noOfFeatures];

            int j;
            for (int i = 0; i < countFeatureVectors; i++) {
                s = br.readLine();
                j = 0;
                while (s.contains(",")) {
                    features[i][j] = s.substring(0, s.indexOf(","));
                    s = s.substring(s.indexOf(",") + 1);
                    j++;
                }
            }
            br.close();
            return features;
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
    }

    public static String[] readExtractedDisorders(String fileName) throws FeatureReadingException{ // expects that the disorder is the final feature
        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            } else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == "") {
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                } else {
                    throw new FeatureReadingException("No features extracted in file " + fileName);
                }
            }
            // the first row should contain features' names, skip it
            int countFeatureVectors = 0;
            s = br.readLine();
            while (s != null) {
                countFeatureVectors++;
                s = br.readLine();
            }
            br.close();

            if (countFeatureVectors == 0){
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                }
                else {
                    throw new FeatureReadingException("No features extracted in file "+fileName);
                }
            }
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            }
            else {
                br = new BufferedReader(new FileReader(fileName));
            }
            s = br.readLine(); // the first row should contain features' names, ignore it
            String [] extractedDisorders = new String[countFeatureVectors];

            int j;
            for (int i = 0; i < countFeatureVectors; i++) {
                s = br.readLine();
                while (s.contains(",")) {
                    s = s.substring(s.indexOf(",") + 1);
                }
                // the final feature is disorder
                extractedDisorders[i] = s.trim();
            }
            br.close();
            return extractedDisorders;
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
    }
    public static double[][] readNumericExtractedFeatures() throws FeatureReadingException{
        return ExtractedFeatures.readNumericExtractedFeatures(null, false);
    }
    public static double[][] readNumericExtractedFeatures(String fileName, boolean includeOrdinalNumber) throws FeatureReadingException {
        // expects that the disorder is the final feature, does not read it
        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            }
            else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == ""){
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                }
                else {
                    throw new FeatureReadingException("No features extracted in file "+fileName);
                }
            }
            // the first row should contain features' names, use it to discover the number of features
            int noOfFeatures = 0;
            while (s.contains(",")){
                s = s.substring(s.indexOf(",")+1);
                noOfFeatures++;
            }
            //noOfFeatures++;
            int countFeatureVectors = 0;
            s = br.readLine();
            while (s != null){
                countFeatureVectors++;
                s = br.readLine();
            }
            br.close();

            if (countFeatureVectors == 0 || noOfFeatures <= 5){
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                }
                else {
                    throw new FeatureReadingException("No features extracted in file "+fileName);
                }
            }
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            }
            else {
                br = new BufferedReader(new FileReader(fileName));
            }
            s = br.readLine(); // the first row should contain features' names, ignore it now
            double [][] features;

            if (includeOrdinalNumber){
                features = new double[countFeatureVectors][noOfFeatures-4];
            }
            else {
                features = new double[countFeatureVectors][noOfFeatures-5]; // note that if additional starting information features are added in the write... method, then this constant needs to be changed
            }
            int j;
            if (includeOrdinalNumber){
                for (int i = 0; i < countFeatureVectors; i++) {
                    s = br.readLine();
                    j = 0;
                    while (s.contains(",")) {
                        if (j == 0) { // note that if additional starting information features are added in the write... method, then this constant needs to be changed
                            features[i][j] = Double.parseDouble(s.substring(0, s.indexOf(",")));
                        }
                        else if (j >= 5){
                            features[i][j-4] = Double.parseDouble(s.substring(0, s.indexOf(",")));
                        }
                        s = s.substring(s.indexOf(",") + 1);
                        j++;
                    }
                }
            }
            else {
                for (int i = 0; i < countFeatureVectors; i++) {
                    s = br.readLine();
                    j = 0;
                    while (s.contains(",")) {
                        if (j >= 5) { // note that if additional starting information features are added in the write... method, then this constant needs to be changed
                            features[i][j-5] = Double.parseDouble(s.substring(0, s.indexOf(",")));
                        }
                        s = s.substring(s.indexOf(",") + 1);
                        j++;
                    }
                }
            }
            br.close();
            return features;
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
    }

    public static double[][] readNumericExtractedFeatures(int [] featureIndices) throws FeatureReadingException{
        return ExtractedFeatures.readNumericExtractedFeatures(null, featureIndices, false);
    }
    /**
     *
     * @param fileName Name of the file from which the features are read
     * @param featureIndices Sorted indices (from smallest to largest index, start at 0) of features that need to be read
     * @return
     * @throws FeatureReadingException
     */
    public static double[][] readNumericExtractedFeatures(String fileName, int [] featureIndices, boolean includeOrdinalNumber) throws FeatureReadingException {
        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            }
            else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == ""){
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                }
                else {
                    throw new FeatureReadingException("No features extracted in file "+fileName);
                }
            }
            // the first row should contain features' names, use it to discover the number of features
            int noOfFeatures = 0;
            while (s.contains(",")){
                s = s.substring(s.indexOf(",")+1);
                noOfFeatures++;
            }
            noOfFeatures++;
            int countFeatureVectors = 0;
            s = br.readLine();
            while (s != null){
                countFeatureVectors++;
                s = br.readLine();
            }
            br.close();

            if (countFeatureVectors == 0 || noOfFeatures <= 5){
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                }
                else {
                    throw new FeatureReadingException("No features extracted in file "+fileName);
                }
            }
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            }
            else {
                br = new BufferedReader(new FileReader(fileName));
            }
            s = br.readLine(); // the first row should contain features' names, ignore it now
            double [][] features;

            if (includeOrdinalNumber){
                features = new double[countFeatureVectors][1+featureIndices.length];
            }
            else {
                features = new double[countFeatureVectors][featureIndices.length];
            }
            int j, k;

            if (includeOrdinalNumber){
                for (int i = 0; i < countFeatureVectors; i++) {
                    s = br.readLine();
                    j = 0;
                    k = 0;
                    while (s.contains(",")) {
                        if (j == 0) {
                            features[i][j] = Double.parseDouble(s.substring(0, s.indexOf(",")));
                            k++;
                        }
                        else if (j >= 5 && featureIndices[k-1] == j - 5) {
                            features[i][k] = Double.parseDouble(s.substring(0, s.indexOf(",")));
                            k++;
                            if (k-1==featureIndices.length)break;
                        }
                        s = s.substring(s.indexOf(",") + 1);
                        j++;
                    }
                }
            }
            else {
                for (int i = 0; i < countFeatureVectors; i++) {
                    s = br.readLine();
                    j = 0;
                    k = 0;
                    while (s.contains(",")) {
                        if (j >= 5 && featureIndices[k] == j - 5) { // note that if additional starting information features are added in the write... method, then this constant needs to be changed
                            features[i][k] = Double.parseDouble(s.substring(0, s.indexOf(",")));
                            k++;
                            if (k==featureIndices.length)break;
                        }
                        s = s.substring(s.indexOf(",") + 1);
                        j++;
                    }
                }
            }
            br.close();
            return features;
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
    }
    public static List<String> readAllExtractedFeaturesNames(String fileName) throws FeatureReadingException{
        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            } else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == "") {
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                } else {
                    throw new FeatureReadingException("No features extracted in file " + fileName);
                }
            }
            // the first row should contain features' names, other rows are ignored
            List<String> featureNames = new ArrayList<>();

            int i = 0;
            while (s.contains(",")) {
                 featureNames.add(s.substring(0, s.indexOf(",")));
                 s = s.substring(s.indexOf(",") + 1);
            }
            featureNames.add(s);
            br.close();
            return featureNames;
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
    }
    public static List<String> readExtractedFeaturesNames() throws FeatureReadingException {
        return ExtractedFeatures.readExtractedFeaturesNames(null, false);
    }
    public static List<String> readExtractedFeaturesNames(String fileName, boolean includeOrdinalNumber) throws FeatureReadingException{
        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            } else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == "") {
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                } else {
                    throw new FeatureReadingException("No features extracted in file " + fileName);
                }
            }
            // the first row should contain features' names, other rows are ignored
            List<String> featureNames = new ArrayList<>();

            if (includeOrdinalNumber){
                int i = 0;
                while (s.contains(",")) {
                    if (i == 0 || i >= 5) {
                        featureNames.add(s.substring(0, s.indexOf(",")));
                    }
                    s = s.substring(s.indexOf(",") + 1);
                    i++;
                }
            }
            else {
                int i = 0;
                while (s.contains(",")) {
                    if (i >= 5) {
                        featureNames.add(s.substring(0, s.indexOf(",")));
                    }
                    s = s.substring(s.indexOf(",") + 1);
                    i++;
                }
            }
            featureNames.add(s);
            br.close();
            return featureNames;
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
    }
    public static List<String> readExtractedFeaturesNames(int [] featureIndices) throws FeatureReadingException{
        return ExtractedFeatures.readExtractedFeaturesNames(null, featureIndices, false, false);
    }
    public static List<String> readExtractedFeaturesNames(String fileName, int [] featureIndices, boolean includeStartingInformationFeatures, boolean includeOrdinalNumber) throws FeatureReadingException{
        String [] featuresNames = null;

        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            } else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == "") {
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                } else {
                    throw new FeatureReadingException("No features extracted in file " + fileName);
                }
            }
            // the first row should contain features' names, other rows are ignored
            List<String> featureNames = new ArrayList<>();

            if (includeStartingInformationFeatures) {
                int i = 0;
                int j = 0;
                while (s.contains(",")) {
                    if (i < 5) {
                        featureNames.add(s.substring(0, s.indexOf(",")));
                    }
                    else if (i >= 5 && i - 5 == featureIndices[j]){
                        featureNames.add(s.substring(0, s.indexOf(",")));
                        j++;
                    }
                    s = s.substring(s.indexOf(",") + 1);
                    i++;
                }
            }
            else if (includeOrdinalNumber){
                int i = 0;
                int j = 0;
                while (s.contains(",")) {
                    if (i == 0) {
                        featureNames.add(s.substring(0, s.indexOf(",")));
                    }
                    else if (i >= 5 && i - 5 == featureIndices[j]){
                        featureNames.add(s.substring(0, s.indexOf(",")));
                        j++;
                    }
                    s = s.substring(s.indexOf(",") + 1);
                    i++;
                }
            }
            else {
                int i = 0;
                int j = 0;
                while (s.contains(",")) {
                    if (i >= 5 && i - 5 == featureIndices[j]) {
                        featureNames.add(s.substring(0, s.indexOf(",")));
                        j++;
                    }
                    s = s.substring(s.indexOf(",") + 1);
                    i++;
                }
            }
            featureNames.add(s);
            br.close();
            return featureNames;
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
    }
    public static int getFeaturesCount(String fileName,  boolean includeStartingInformationFeatures, boolean includeOrdinalNumber) throws FeatureReadingException{
        int count = 0;

        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            } else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == "") {
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                } else {
                    throw new FeatureReadingException("No features extracted in file " + fileName);
                }
            }
            // the first row should contain features' names, other rows are ignored

            int i = 0;

            if (includeStartingInformationFeatures){
                while(s.contains(",")){
                    s = s.substring(s.indexOf(",")+1);
                    count++;
                }
                count++;
            }
            else if (includeOrdinalNumber){
                if (s.contains(",")){
                    s = s.substring(s.indexOf(",")+1);
                    count++;
                }
                i = 0;
                while(s.contains(",")){
                    if (i == 4){
                        break;
                    }
                    s = s.substring(s.indexOf(",")+1);
                    i++;
                }
                while(s.contains(",")){
                    s = s.substring(s.indexOf(",")+1);
                    count++;
                }
                count++;
            }
            else {             // skip the first five information features
                i = 0;
                while(s.contains(",")){
                    if (i == 5){
                        break;
                    }
                    s = s.substring(s.indexOf(",")+1);
                    i++;
                }
                while(s.contains(",")){
                    s = s.substring(s.indexOf(",")+1);
                    count++;
                }
                count++;
            }
            br.close();
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
        return count;
    }
    /**
     *
     * @param fileName
     * @param featuresNames Warning: this method assumes that featuresNames are unique, otherwise it won't work correctly
     * @return
     * @throws FeatureReadingException
     */
    public static int[] getFeaturesIndicesForFeaturesNames(String fileName, String [] featuresNames) throws FeatureReadingException {
        int[] featuresIndices = null;

        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            } else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == "") {
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                } else {
                    throw new FeatureReadingException("No features extracted in file " + fileName);
                }
            }
            // the first row should contain features' names, other rows are ignored
            featuresIndices = new int[featuresNames.length];
            // skip the first five information features
            int i = 0;
            String temp;
            while(s.contains(",")){
                if (i == 5){
                    i = 0;
                    break;
                }
                s = s.substring(s.indexOf(",")+1);
                i++;
            }
            while(s.contains(",")){
                temp = s.substring(0,s.indexOf(","));
                for (int j = 0; j < featuresNames.length; j++){
                    if (temp.equals(featuresNames[j])){
                        featuresIndices[j] = i;
                        break;
                    }
                }
                s = s.substring(s.indexOf(",")+1);
                i++;
            }
            br.close();
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
        return featuresIndices;
    }

            /**
             *
             * @param fileName File name containing the extracted feature vectors
             * @param last if true, last variable is goal variable and goalIndex is ignored
             * @param goalIndex if last is false, then goalIndex will be read
             * @return
             */
    public static int[] readClassesFromFeatureVectorsFileAsIntegers(String fileName, boolean last, int goalIndex) throws FeatureReadingException{
        int [] classes = null;

        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            } else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == "") {
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                } else {
                    throw new FeatureReadingException("No features extracted in file " + fileName);
                }
            }
            // the first row should contain features' names, use it to discover the number of features
            int noOfFeatures = 0;
            while (s.contains(",")){
                s = s.substring(s.indexOf(",")+1);
                noOfFeatures++;
            }
            noOfFeatures++;
            int countFeatureVectors = 0;
            s = br.readLine();
            while (s != null){
                countFeatureVectors++;
                s = br.readLine();
            }
            br.close();

            if (countFeatureVectors == 0 || noOfFeatures <= 5){
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                }
                else {
                    throw new FeatureReadingException("No features extracted in file "+fileName);
                }
            }
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            }
            else {
                br = new BufferedReader(new FileReader(fileName));
            }
            s = br.readLine(); // the first row should contain features' names, ignore it now

            classes = new int[countFeatureVectors];
            int j;
            if (last) {
                for (int i = 0; i < countFeatureVectors; i++) {
                    s = br.readLine();
                    classes[i] = Integer.parseInt(s.substring(s.lastIndexOf(",")+1));
                }
            }
            else {
                for (int i = 0; i < countFeatureVectors; i++) {
                    s = br.readLine();
                    j=0;
                    while (s.contains(",")) {
                        if (goalIndex == j) {
                            classes[i] = Integer.parseInt(s.substring(0,s.indexOf(",")));
                            break;
                        }
                        else {
                            s = s.substring(s.indexOf(",")+1);
                            j++;
                        }
                    }
                }
            }
            br.close();
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
        catch (Exception exc){
            throw new FeatureReadingException("Classes are not integer numbers");
        }
        return classes;
    }
    public static String[] readClassesFromFeatureVectorsFileAsStrings(String fileName, boolean last, int goalIndex) throws FeatureReadingException{
        String [] classes = null;

        try {
            BufferedReader br = null;
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            } else {
                br = new BufferedReader(new FileReader(fileName));
            }
            String s = br.readLine();
            if (s == null || s == "") {
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                } else {
                    throw new FeatureReadingException("No features extracted in file " + fileName);
                }
            }
            // the first row should contain features' names, use it to discover the number of features
            int noOfFeatures = 0;
            while (s.contains(",")){
                s = s.substring(s.indexOf(",")+1);
                noOfFeatures++;
            }
            noOfFeatures++;
            int countFeatureVectors = 0;
            s = br.readLine();
            while (s != null){
                countFeatureVectors++;
                s = br.readLine();
            }
            br.close();

            if (countFeatureVectors == 0 || noOfFeatures <= 5){
                if (fileName == null) {
                    throw new FeatureReadingException("No features extracted in file Outputfilea.csv");
                }
                else {
                    throw new FeatureReadingException("No features extracted in file "+fileName);
                }
            }
            if (fileName == null) {
                br = new BufferedReader(new FileReader("Outputfilea.csv"));
            }
            else {
                br = new BufferedReader(new FileReader(fileName));
            }
            s = br.readLine(); // the first row should contain features' names, ignore it now

            classes = new String[countFeatureVectors];
            int j;
            if (last) {
                for (int i = 0; i < countFeatureVectors; i++) {
                    s = br.readLine();
                    classes[i] = s.substring(s.lastIndexOf(",")+1);
                }
            }
            else {
                for (int i = 0; i < countFeatureVectors; i++) {
                    s = br.readLine();
                    j=0;
                    while (s.contains(",")) {
                        if (goalIndex == j) {
                            classes[i] = s.substring(0,s.indexOf(","));
                            break;
                        }
                        else {
                            s = s.substring(s.indexOf(",")+1);
                            j++;
                        }
                    }
                }
            }
            br.close();
        }
        catch (IOException exc){
            throw new FeatureReadingException(exc.getMessage());
        }
        catch (Exception exc){
            throw new FeatureReadingException("Classes are not integer numbers");
        }
        return classes;
    }
    public static int readNumberOfClassesFromFeatureVectorsFile(String fileName, boolean last, int goalIndex) throws FeatureReadingException {
        int countClasses = 0;

        String[] classes = ExtractedFeatures.readClassesFromFeatureVectorsFileAsStrings(fileName, last, goalIndex);

        List<String> differentClasses = new ArrayList<>();

        for (int i = 0; i < classes.length; i++) {
            if (!differentClasses.contains(classes[i])) {
                differentClasses.add(classes[i]);
            }
        }
        return differentClasses.size();
    }
}
