package multisab.processing.eegAnalysis;

import multisab.processing.commonSignalFeatures.timeDomain.statisticMeasure.Statistics;

//time domain features taken from: X.-W. Wang, D. Nie, and B.-L. Lu, “EEG-Based Emotion Recognition Using Frequency Domain ExtractedFeatures and Support Vector Machines,” Lecture Notes in Computer Science, vol. 7062, pp. 734–743, 2011.

public class TimeDomainAnalysisEEG {
    /**
     * Calculates mean of first differences normalized by standard deviation
     *
     * @param segment
     * @return
     */
    public static final double calculateMeanOfFirstDiffNormalized(double[] segment) {
        double firstDiff = Statistics.meanOfAbsoluteValuesOfFirstDifferences(segment);
        double stdDev = Statistics.standardDeviation(segment);
        return firstDiff / stdDev;
    }

    /**
     * Calculates mean of first differences normalized by standard deviation from the beginning interval to the ending interval
     *
     * @param segment
     * @param intervalBeginning
     * @param intervalEnding
     * @return
     */
    public static final double calculateMeanOfFirstDiffNormalized(double[] segment, int intervalBeginning, int intervalEnding) {
        double firstDiff = Statistics.meanOfAbsoluteValuesOfFirstDifferences(segment, intervalBeginning, intervalEnding);
        double stdDev = Statistics.standardDeviation(segment, intervalBeginning, intervalEnding);
        return firstDiff / stdDev;
    }

    /**
     * Calculates mean of second differences normalized by standard deviation
     *
     * @param segment
     * @return
     */
    public static final double calculateMeanOfSecondDiffNormalized(double[] segment) {
        double secondDiff = Statistics.meanOfAbsoluteValuesOfSecondDifferences(segment);
        double stdDev = Statistics.standardDeviation(segment);
        return secondDiff / stdDev;
    }

    /**
     * Calculates mean of second differences normalized by standard deviation from the beginning interval to the ending interval
     *
     * @param segment
     * @param intervalBeginning
     * @param intervalEnding
     * @return
     */
    public static final double calculateMeanOfSecondDiffNormalized(double[] segment, int intervalBeginning, int intervalEnding) {
        double secondDiff = Statistics.meanOfAbsoluteValuesOfSecondDifferences(segment, intervalBeginning, intervalEnding);
        double stdDev = Statistics.standardDeviation(segment, intervalBeginning, intervalEnding);
        return secondDiff / stdDev;
    }

    /**
     * Calculates mean of first differences
     *
     * @param segment
     * @return
     */
    public static final double calculateMeanOfFirstDiff(double[] segment) {
        return Statistics.meanOfAbsoluteValuesOfFirstDifferences(segment);
    }

    /**
     * Calculates mean of first differences from the beginning interval to the ending interval
     *
     * @param segment
     * @param intervalBeginning
     * @param intervalEnding
     * @return
     */
    public static final double calculateMeanOfFirstDiff(double[] segment, int intervalBeginning, int intervalEnding) {
        return Statistics.meanOfAbsoluteValuesOfFirstDifferences(segment, intervalBeginning, intervalEnding);
    }

    /**
     * Calculates mean of second differences
     *
     * @param segment
     * @return
     */
    public static final double calculateMeanOfSecondDiff(double[] segment) {
        return Statistics.meanOfAbsoluteValuesOfSecondDifferences(segment);
    }

    /**
     * Calculates mean of second differences from the beginning interval to the ending interval
     *
     * @param segment
     * @param intervalBeginning
     * @param intervalEnding
     * @return
     */
    public static final double calculateMeanOfSecondDiff(double[] segment, int intervalBeginning, int intervalEnding) {
        return Statistics.meanOfAbsoluteValuesOfSecondDifferences(segment, intervalBeginning, intervalEnding);
    }

    /**
     * Calculates mean value of a segment
     *
     * @param segment
     * @return
     */
    public static final double calculateMean(double[] segment) {
        return Statistics.mean(segment);
    }

    /**
     * Calculates mean value of a segment from the beginning interval to the ending interval
     *
     * @param segment
     * @param intervalBeginning
     * @param intervalEnding
     * @return
     */
    public static final double calculateMean(double[] segment, int intervalBeginning, int intervalEnding) {
        return Statistics.mean(segment, intervalBeginning, intervalEnding);
    }

    /**
     * Calculates standard deviation value of a segment
     *
     * @param segment
     * @return
     */
    public static final double calculateStandardDeviation(double[] segment) {
        return Statistics.standardDeviation(segment);
    }

    /**
     * Calculates standard deviation value of a segment from the beginning interval to the ending interval
     *
     * @param segment
     * @param intervalBeginning
     * @param intervalEnding
     * @return
     */
    public static final double calculateStandardDeviation(double[] segment, int intervalBeginning, int intervalEnding) {
        return Statistics.standardDeviation(segment, intervalBeginning, intervalEnding);
    }
}
