package multisab.processing.hrvAnalysis;

import multisab.processing.commonSignalFeatures.frequencyDomain.SpectralAnalysis;
import multisab.processing.multisabException.ProcessingException;

public class SpectralAnalysisHRV {
    public static final double DEFAULT_ULTRA_LOW_FREQUENCY_LOWER_BOUND = 0.00001;
    public static final double DEFAULT_ULTRA_LOW_FREQUENCY_UPPER_BOUND = 0.003;
    public static final double DEFAULT_VERY_LOW_FREQUENCY_LOWER_BOUND = 0.003;
    public static final double DEFAULT_VERY_LOW_FREQUENCY_UPPER_BOUND = 0.04;
    public static final double DEFAULT_LOW_FREQUENCY_LOWER_BOUND = 0.04;
    public static final double DEFAULT_LOW_FREQUENCY_UPPER_BOUND = 0.15;
    public static final double DEFAULT_HIGH_FREQUENCY_LOWER_BOUND = 0.15;
    public static final double DEFAULT_HIGH_FREQUENCY_UPPER_BOUND = 0.4;
    public static final int DEFAULT_ORDER = 12;
    public static final int FAST_FOURIER_TRANSFORM = 1;
    public static final int BURG_METHOD = 2;
    public static final int LOMB_SCARGLE_METHOD = 3;
    public static final String NO_WINDOW = "NO_WINDOW";
    public static final String HAMMING_WINDOW = "HAMMING_WINDOW";
    public static final String HANN_WINDOW = "HANN_WINDOW";
    public static final double DEFAULT_SAMPLING_FREQUENCY = 1.0;
    SpectralAnalysis spectHRV = null;
    double[] frequencies;
    double[] psdEstimate;
        /*
         * Spectral entropy estimation based on:
	    * Ermes, M., Pärkkä, J. & Cluitmans, L. 2008. Advancing from Offline to
	    * Online Activity Recognition with Wearable Sensors. Proceedings of the
		* 30th Annual International Conference of the IEEE Engineering in Medicine
		* and Biology Society. Pp. 4451–4454. 
		* 
		*  A. Rezek and S. J. Roberts, “Stochastic Complexity Measures for Physiological Signal Analysis,” IEEE Trans. Biomed. Eng., vol. 45, no. 9, pp. 1186–1191, Sep. 1998.
		*
		* The original paper uses fast Fourier transform to obtain spectrum and calculate the entropy.
		* This method may be used to calculate Spectral entropy using any previously obtained PSD estimate
	    * @param lowerFrequencyBound Lower bound for spectral entropy
		* @param upperFrequencyBound Higher bound for spectral entropy
		* @return spectral entropy
		*/

    public SpectralAnalysisHRV(double[] segment, double sampFreq, int method, String window, int order) throws ProcessingException {
        spectHRV = new SpectralAnalysis(segment, sampFreq);
        boolean success = false;
        switch (method) {
            case SpectralAnalysis.FAST_FOURIER_TRANSFORM:
                success = spectHRV.calculateSpectrumFourier(window);
                break;
            case SpectralAnalysis.BURG_METHOD:
                success = spectHRV.calculateSpectrumBurg(order);
                break;
            case SpectralAnalysis.LOMB_SCARGLE_METHOD:
                success = spectHRV.calculateSpectrumLombScargle();
                break;
        }
        if (!success) throw new ProcessingException("Unable to calculate HRV spectrum.");
        frequencies = spectHRV.getFrequencies();
        psdEstimate = spectHRV.getPSDEstimate();
    }
	/*
 	* Spectral entropy estimation based on:
    * Ermes, M., Pärkkä, J. & Cluitmans, L. 2008. Advancing from Offline to
    * Online Activity Recognition with Wearable Sensors. Proceedings of the
	* 30th Annual International Conference of the IEEE Engineering in Medicine
	* and Biology Society. Pp. 4451–4454. 
	* 
	*  A. Rezek and S. J. Roberts, “Stochastic Complexity Measures for Physiological Signal Analysis,” IEEE Trans. Biomed. Eng., vol. 45, no. 9, pp. 1186–1191, Sep. 1998.
	*
	* The original paper uses fast Fourier transform to obtain spectrum and calculate the entropy.
	* This method may be used to calculate Spectral entropy using any previously obtained PSD estimate
    * @return spectral entropy
	*/

    public double getTotalPSD() {
        double sum = 0.0;
        for (int i = 0; i < this.frequencies.length; i++) {
            sum += psdEstimate[i];
        }
        return sum;
    }

    /**
     * A method for getting PSD within a defined frequency band
     *
     * @param lowerFrequency Lower bound of the band
     * @param upperFrequency Upper bound of the band
     * @return Power spectral density estimate
     */
    public double getPSDForFrequencyBand(double lowerFrequency, double upperFrequency) {
        double sum = 0.0;
        int minIndex = 0;
        int maxIndex = 0;
        int i;
        for (i = 0; i < this.frequencies.length; i++) {
            if (frequencies[i] >= lowerFrequency) {
                break;
            }
        }
        minIndex = i;
        for (i = minIndex; i < this.frequencies.length; i++) {
            if (frequencies[i] > upperFrequency) {
                break;
            }
        }
        maxIndex = i - 1;

        for (i = minIndex; i <= maxIndex; i++) {
            sum += psdEstimate[i];
        }
        return sum;
    }

    public double getFrequencyBand(double lowerFrequency, double upperFrequency) {
        double sum = 0.0;
        int minIndex = 0;
        int maxIndex = 0;
        int i;
        for (i = 0; i < this.frequencies.length; i++) {
            if (frequencies[i] >= lowerFrequency) {
                break;
            }
        }
        minIndex = i;
        for (i = minIndex; i < this.frequencies.length; i++) {
            if (frequencies[i] > upperFrequency) {
                break;
            }
        }
        maxIndex = i;

        double[] bandFrequencies = new double[maxIndex - minIndex];
        for (i = minIndex; i < maxIndex; i++) {
            bandFrequencies[i] = this.frequencies[i];
        }
        return sum;
    }

    /**
     * @return Ultra low frequency PSD component (<0.003 Hz)
     */
    public double getULF() {
        return this.getPSDForFrequencyBand(DEFAULT_ULTRA_LOW_FREQUENCY_LOWER_BOUND, DEFAULT_ULTRA_LOW_FREQUENCY_UPPER_BOUND);
    }

    /**
     * Method for obtaining ultra low frequency PSD component within defined frequency band
     *
     * @param ultraLowFrequencyLowerBound lower bound of the ULF band
     * @param ultraLowFrequencyUpperBound upper bound of the ULF band
     * @return Ultra low frequency PSD component
     */
    public double getULF(double ultraLowFrequencyLowerBound, double ultraLowFrequencyUpperBound) {
        return this.getPSDForFrequencyBand(ultraLowFrequencyLowerBound, ultraLowFrequencyUpperBound);
    }

    /**
     * @return Very low frequency PSD component ([0.003,0.04] Hz)
     */
    public double getVLF() {
        return this.getPSDForFrequencyBand(DEFAULT_VERY_LOW_FREQUENCY_LOWER_BOUND, DEFAULT_VERY_LOW_FREQUENCY_UPPER_BOUND);
    }

    /**
     * Method for obtaining very low frequency PSD component within defined frequency band
     *
     * @param veryLowFrequencyLowerBound lower bound of the VLF band
     * @param veryLowFrequencyUpperBound upper bound of the VLF band
     * @return Very low frequency PSD component
     */
    public double getVLF(double veryLowFrequencyLowerBound, double veryLowFrequencyUpperBound) {
        return this.getPSDForFrequencyBand(veryLowFrequencyLowerBound, veryLowFrequencyUpperBound);
    }

    /**
     * @return Low frequency PSD component ([0.04,0.15] Hz)
     */
    public double getLF() {
        return this.getPSDForFrequencyBand(DEFAULT_LOW_FREQUENCY_LOWER_BOUND, DEFAULT_LOW_FREQUENCY_UPPER_BOUND);
    }

    /**
     * Method for obtaining low frequency PSD component within defined frequency band
     *
     * @param lowFrequencyLowerBound lower bound of the LF band
     * @param lowFrequencyUpperBound upper bound of the LF band
     * @return Low frequency PSD component
     */
    public double getLF(double lowFrequencyLowerBound, double lowFrequencyUpperBound) {
        return this.getPSDForFrequencyBand(lowFrequencyLowerBound, lowFrequencyUpperBound);
    }

    /**
     * @return High frequency PSD component ([0.14,0.4] Hz)
     */
    public double getHF() {
        return this.getPSDForFrequencyBand(DEFAULT_HIGH_FREQUENCY_LOWER_BOUND, DEFAULT_HIGH_FREQUENCY_UPPER_BOUND);
    }

    /**
     * Method for obtaining high frequency PSD component within defined frequency band
     *
     * @param highFrequencyLowerBound lower bound of the HF band
     * @param highFrequencyUpperBound upper bound of the HF band
     * @return High frequency PSD component
     */
    public double getHF(double highFrequencyLowerBound, double highFrequencyUpperBound) {
        return this.getPSDForFrequencyBand(highFrequencyLowerBound, highFrequencyUpperBound);
    }

    /**
     * @return Ratio of LF and HF spectral components, reflecting autonomic system response
     */
    public double getLFHFRatio() {
        return this.getLF() / this.getHF();
    }

    /**
     * Ratio of LF and HF spectral components, within a defined frequency band
     *
     * @param lowFrequencyLowerBound  lower bound of the LF band
     * @param lowFrequencyUpperBound  upper bound of the LF band
     * @param highFrequencyLowerBound lower bound of the HF band
     * @param highFrequencyUpperBound upper bound of the HF band
     * @return Ratio of LF and HF spectral components, reflecting autonomic system response
     */
    public double getLFHFRatio(double lowFrequencyLowerBound, double lowFrequencyUpperBound, double highFrequencyLowerBound, double highFrequencyUpperBound) {
        return this.getLF(lowFrequencyLowerBound, lowFrequencyUpperBound) / this.getHF(highFrequencyLowerBound, highFrequencyUpperBound);
    }

    /**
     * This method may be used to calculate Spectral entropy using any previously obtained PSD estimate
     *
     * @param lowerFrequencyBound Lower bound for spectral entropy
     * @param upperFrequencyBound Higher bound for spectral entropy
     * @return
     */
    public double getSpectralEntropy(double lowerFrequencyBound, double upperFrequencyBound) {
        double spectEntropy = 0;
        int i;

        int minIndex = 0, maxIndex = 0;
        for (i = 0; i < frequencies.length; i++) {
            if (frequencies[i] >= lowerFrequencyBound) {
                break;
            }
        }
        minIndex = i;

        for (i = minIndex; i < frequencies.length; i++) {
            if (frequencies[i] > upperFrequencyBound) {
                break;
            }
        }
        maxIndex = i - 1;

        for (i = minIndex; i <= maxIndex; i++) {
            spectEntropy += psdEstimate[i] * Math.log10(frequencies[i]);
        }
        spectEntropy = -spectEntropy / Math.log10((double) (maxIndex - minIndex));

        return spectEntropy;
    }

    /**
     * This method may be used to calculate Spectral entropy using any previously obtained PSD estimate
     *
     * @return
     */
    public double getSpectralEntropy() {
        double spectEntropy = 0;
        int i;

        for (i = 0; i < frequencies.length; i++) {
            spectEntropy += psdEstimate[i] * Math.log10(frequencies[i]);
        }
        spectEntropy = -spectEntropy / Math.log10((double) (frequencies.length));

        return spectEntropy;
    }
}

