package multisab.processing.commonSignalFeatures.nonlinear.entropy;

import multisab.processing.commonSignalFeatures.timeDomain.statisticMeasure.Statistics;

import java.util.Arrays;

public class AlphabetEntropy {
    /**
     * Default threshold for detecting a change in the signal
     */
    public static final double NO_CHANGE_THRESHOLD_DEFAULT = 1e-03 + 1e-07;
    public static final int MINIMAL_LENGTH_FOR_EXTRACTION = 4;
    /**
     * Size of the alphabet
     */
    private static final int ALPHABET_SIZE = 27;
    /**
     * Coded alphabet of the signal
     */
    private int[] codedAlphabet = null;
    /**
     * Number of individual letters in the signal
     */
    private int[] countAlphabet = null;
    /**
     * Rate of individual letters in the signal - number of individual letters divided by segment length
     */
    private double[] rateAlphabet = null;
    /**
     * Whether letters exist in the signal
     */
    private boolean[] existsLetter = null;
    /**
     * Mean of the 1D expanded alphabet tessellation entropy of the signal
     */
    private double averageAlphabetEntropy = 0.0;
    /**
     * Variance of the 1D expanded alphabet tessellation entropy of the signal
     */
    private double varianceAlphabetEntropy = 0.0;
    /**
     * Maximum of the 1D expanded alphabet tessellation entropy of the signal
     */
    private double maximumAlphabetEntropy = 0.0;
    /**
     * Average of the 1D expanded alphabet tessellation entropy per letter of the signal
     */
    private double[] averageAlphabetEntropyPerLetter = null;
    /**
     * Maximum of the 1D expanded alphabet tessellation entropy per letter of the signal
     */
    private double[] maximumAlphabetEntropyPerLetter = null;
    /**
     * Variance of the 1D expanded alphabet tessellation entropy per letter of the signal
     */
    private double[] varianceAlphabetEntropyPerLetter = null;
    /**
     * Threshold for detecting a change in the signal
     */
    private double observationUnchangedTime = AlphabetEntropy.NO_CHANGE_THRESHOLD_DEFAULT;
    /**
     * Whether the entropy can be calculated
     */
    private boolean calculated = false;
    /**
     * Whether the array is to small to calculate the entropy
     */
    private boolean tooSmallArray = false;
    /**
     * Local variable for storing a single alphabetEntropy calculation
     */
    private double alphabetEntropy;

    /**
     * A light-weight constructor intended for consecutive single calculations of Alphabet entropy
     *
     * @param observationUnchangedTime Threshold for detecting a change in the signal, in milliseconds
     */
    public AlphabetEntropy(double observationUnchangedTime) {
        this.observationUnchangedTime = 0.001 * observationUnchangedTime + 1e-07;
    }

    /**
     * Constructor takes a series of points and calculates its expanded alphabet 1D tessellation entropy
     *
     * @param series                   Series of points, usually RR intervals
     * @param observationUnchangedTime Threshold for detecting a change in the signal, in milliseconds
     */
    public AlphabetEntropy(double[] series, double observationUnchangedTime, boolean singleCalculation) {
        this.observationUnchangedTime = 0.001 * observationUnchangedTime + 1e-07;
        codedAlphabet = this.codeDataECG(series);
        if (!singleCalculation) {
            if (!tooSmallArray) {
                countAlphabet = this.countAlphabet(codedAlphabet);
                rateAlphabet = new double[countAlphabet.length];
                for (int i = 0; i < rateAlphabet.length; i++) {
                    rateAlphabet[i] = (double) (countAlphabet[i]) / (series.length - 3);
                }
                existsLetter = this.existsLetter(codedAlphabet);
                calculated = this.calculateExpandedTessellationEntropy(series, codedAlphabet);
            } else calculated = false;
        } else { // single calculation
            if (codedAlphabet != null) {
                alphabetEntropy = this.calculateExpandedTessellationEntropySingleLetter(series, codedAlphabet[0]);
            } else {
                calculated = false;
            }
        }
    }

    /**
     * Encodes a single letter for a series of 4 consecutive measurements.
     *
     * @param series A series of 4 measurements
     * @return A single letter as int
     */
    public int encodeSingleSequence(double[] series) {
        int[] encodedLetters = this.codeDataECG(series);
        if (encodedLetters != null) {
            return encodedLetters[0];
        } else return -1;
    }

    /**
     * Return a letter representation in int as Alphabet entropy coded string
     *
     * @param letter
     * @return
     */
    public static String letterToString(int letter) {
        switch (letter) {
            case 0:
                return "A";
            case 1:
                return "B";
            case 2:
                return "C";
            case 3:
                return "D";
            case 4:
                return "E";
            case 5:
                return "F";
            case 6:
                return "G";
            case 7:
                return "H";
            case 8:
                return "I";
            case 9:
                return "J";
            case 10:
                return "K";
            case 11:
                return "L";
            case 12:
                return "M";
            case 13:
                return "N";
            case 14:
                return "O";
            case 15:
                return "P";
            case 16:
                return "Q";
            case 17:
                return "R";
            case 18:
                return "S";
            case 19:
                return "T";
            case 20:
                return "U";
            case 21:
                return "V";
            case 22:
                return "W";
            case 23:
                return "X";
            case 24:
                return "Y";
            case 25:
                return "Z";
            case 26:
                return "AA";
            default:
                return "";
        }
    }

    /**
     * @return Coded alphabet for a segment
     */
    public int[] getCodedAlphabet() {
        return this.codedAlphabet;
    }

    /**
     * @return
     */
    public double getSingleAlphabetEntropy() {
        return this.alphabetEntropy;
    }

    /**
     * @return Number of individual letters in the signal
     */
    public int[] getCountAlphabet() {
        return this.countAlphabet;
    }

    /**
     * @return Rate of individual letters in the signal
     */
    public double[] getRateAlphabet() {
        return this.rateAlphabet;
    }

    /**
     * @return Whether individual letters exist in the signal
     */
    public boolean[] getExistsLetter() {
        return this.existsLetter;
    }

    /**
     * @return Whether the entropy has been calculated
     */
    public boolean isCalculated() {
        return this.calculated;
    }

    /**
     * @return Mean of the 1D expanded alphabet tessellation entropy of the signal
     */
    public double getAverageAlphabetEntropy() {
        return this.averageAlphabetEntropy;
    }

    /**
     * @return Variance of the 1D expanded alphabet tessellation entropy of the signal
     */
    public double getVarianceAlphabetEntropy() {
        return this.varianceAlphabetEntropy;
    }

    /**
     * @return Maximum of the 1D expanded alphabet tessellation entropy of the signal
     */
    public double getMaximumAlphabetEntropy() {
        return this.maximumAlphabetEntropy;
    }

    /**
     * @return Average of the 1D expanded alphabet tessellation entropy per letter of the signal
     */
    public double[] getAverageAlphabetEntropyPerLetter() {
        return this.averageAlphabetEntropyPerLetter;
    }

    /**
     * @return Variance of the 1D expanded alphabet tessellation entropy per letter of the signal
     */
    public double[] getVarianceAlphabetEntropyPerLetter() {
        return this.varianceAlphabetEntropyPerLetter;
    }

    /**
     * @return Maximum of the 1D expanded alphabet tessellation entropy per letter of the signal
     */
    public double[] getMaximumAlphabetEntropyPerLetter() {
        return this.maximumAlphabetEntropyPerLetter;
    }

    /**
     * @return Threshold for detecting a change in the signal
     */
    public double getObservationUnchangedTime() {
        return this.observationUnchangedTime;
    }

    /**
     * @param observationUnchangedTime Sets threshold for detecting a change in the signal
     */
    public void setObservationUnchangedTime(double observationUnchangedTime) {
        this.observationUnchangedTime = observationUnchangedTime;
    }

    /**
     * Codes data to alphabet:
     * <p>
     * default for 0 (no change) is defined if the length of consecutive RR intervals do not differ by more than 1 ms
     * <p>
     * 000 = a = 0
     * 00+ = b = 1
     * 00- = c = 2
     * 0+0 = d = 3
     * 0-0 = e = 4
     * 0++ = f = 5
     * 0-- = g = 6
     * 0+- = h = 7
     * 0-+ = i = 8
     * +00 = j = 9
     * -00 = k = 10
     * +0+ = l = 11
     * +0- = m = 12
     * -0- = n = 13
     * -0+ = o = 14
     * ++0 = p = 15
     * +-0 = q = 16
     * --0 = r = 17
     * -+0 = s = 18
     * +++ = t = 19
     * ++- = u = 20
     * +-+ = v = 21
     * +-- = w = 22
     * --- = x = 23
     * --+ = y = 24
     * -+- = z = 25
     * -++ = aa = 26
     *
     * @param data A matrix whose rows should be coded by alphabet entropy
     * @return Matrix coded by alphabet entropy
     */
    public int[][] codeData(double[][] data) {
        int M = data.length;
        int N = data[0].length;

        if (N < AlphabetEntropy.MINIMAL_LENGTH_FOR_EXTRACTION) {
            this.tooSmallArray = true;
            return null;
        }

        int[][] alfabetCoded = new int[M][N - 3];

        for (int i = 0; i < M; i++) {
            for (int j = 0; j < N - 3; j++) {
                if (Math.abs(data[i][j + 1] - data[i][j]) <= observationUnchangedTime) {    // first is 0
                    if (Math.abs(data[i][j + 2] - data[i][j + 1]) <= observationUnchangedTime) { // second is 0
                        if (Math.abs(data[i][j + 3] - data[i][j + 2]) <= observationUnchangedTime) { // third is 0
                            alfabetCoded[i][j] = 0;
                        } else if (data[i][j + 3] - data[i][j + 2] > observationUnchangedTime) { // third is +
                            alfabetCoded[i][j] = 1;
                        } else {    // treci je -
                            alfabetCoded[i][j] = 2;
                        }
                    } else if (data[i][j + 2] - data[i][j + 1] > observationUnchangedTime) { // second is +
                        if (Math.abs(data[i][j + 3] - data[i][j + 2]) <= observationUnchangedTime) { // third is 0
                            alfabetCoded[i][j] = 3;
                        } else if (data[i][j + 3] - data[i][j + 2] > observationUnchangedTime) { // third is +
                            alfabetCoded[i][j] = 5;
                        } else {    // treci je -
                            alfabetCoded[i][j] = 7;
                        }
                    } else {    // drugi je -
                        if (Math.abs(data[i][j + 3] - data[i][j + 2]) <= observationUnchangedTime) { // third is 0
                            alfabetCoded[i][j] = 4;
                        } else if (data[i][j + 3] - data[i][j + 2] > observationUnchangedTime) { // third is +
                            alfabetCoded[i][j] = 8;
                        } else {    // treci je -
                            alfabetCoded[i][j] = 6;
                        }
                    }
                } else if (data[i][j + 1] - data[i][j] > observationUnchangedTime) { // first is +
                    if (Math.abs(data[i][j + 2] - data[i][j + 1]) <= observationUnchangedTime) { // second is 0
                        if (Math.abs(data[i][j + 3] - data[i][j + 2]) <= observationUnchangedTime) { // third is 0
                            alfabetCoded[i][j] = 9;
                        } else if (data[i][j + 3] - data[i][j + 2] > observationUnchangedTime) { // third is +
                            alfabetCoded[i][j] = 11;
                        } else {    // treci je -
                            alfabetCoded[i][j] = 12;
                        }
                    } else if (data[i][j + 2] - data[i][j + 1] > observationUnchangedTime) { // second is +
                        if (Math.abs(data[i][j + 3] - data[i][j + 2]) <= observationUnchangedTime) { // third is 0
                            alfabetCoded[i][j] = 15;
                        } else if (data[i][j + 3] - data[i][j + 2] > observationUnchangedTime) { // third is +
                            alfabetCoded[i][j] = 19;
                        } else {    // treci je -
                            alfabetCoded[i][j] = 20;
                        }
                    } else {    // drugi je -
                        if (Math.abs(data[i][j + 3] - data[i][j + 2]) <= observationUnchangedTime) { // third is 0
                            alfabetCoded[i][j] = 16;
                        } else if (data[i][j + 3] - data[i][j + 2] > observationUnchangedTime) { // third is +
                            alfabetCoded[i][j] = 21;
                        } else {    // treci je -
                            alfabetCoded[i][j] = 22;
                        }
                    }
                } else { // prvi je -
                    if (Math.abs(data[i][j + 2] - data[i][j + 1]) <= observationUnchangedTime) { // second is 0
                        if (Math.abs(data[i][j + 3] - data[i][j + 2]) <= observationUnchangedTime) { // third is 0
                            alfabetCoded[i][j] = 10;    // k
                        } else if (data[i][j + 3] - data[i][j + 2] > observationUnchangedTime) { // third is +
                            alfabetCoded[i][j] = 14;    // o
                        } else {    // treci je -
                            alfabetCoded[i][j] = 13;    // n
                        }
                    } else if (data[i][j + 2] - data[i][j + 1] > observationUnchangedTime) { // second is +
                        if (Math.abs(data[i][j + 3] - data[i][j + 2]) <= observationUnchangedTime) { // third is 0
                            alfabetCoded[i][j] = 18;    // s
                        } else if (data[i][j + 3] - data[i][j + 2] > observationUnchangedTime) { // third is +
                            alfabetCoded[i][j] = 26;    // aa
                        } else {    // treci je -
                            alfabetCoded[i][j] = 25;    // z
                        }
                    } else {    // drugi je -
                        if (Math.abs(data[i][j + 3] - data[i][j + 2]) <= observationUnchangedTime) { // third is 0
                            alfabetCoded[i][j] = 17;    // r
                        } else if (data[i][j + 3] - data[i][j + 2] > observationUnchangedTime) { // third is +
                            alfabetCoded[i][j] = 24;    // y
                        } else {    // treci je -
                            alfabetCoded[i][j] = 23;    // x
                        }
                    }
                }
            }
        }
        return alfabetCoded;
    }

    /**
     * Codes data to alphabet:
     * <p>
     * default for 0 (no change) is defined if the length of consecutive RR intervals do not differ by more than 1 ms
     * <p>
     * 000 = a = 0
     * 00+ = b = 1
     * 00- = c = 2
     * 0+0 = d = 3
     * 0-0 = e = 4
     * 0++ = f = 5
     * 0-- = g = 6
     * 0+- = h = 7
     * 0-+ = i = 8
     * +00 = j = 9
     * -00 = k = 10
     * +0+ = l = 11
     * +0- = m = 12
     * -0- = n = 13
     * -0+ = o = 14
     * ++0 = p = 15
     * +-0 = q = 16
     * --0 = r = 17
     * -+0 = s = 18
     * +++ = t = 19
     * ++- = u = 20
     * +-+ = v = 21
     * +-- = w = 22
     * --- = x = 23
     * --+ = y = 24
     * -+- = z = 25
     * -++ = aa = 26
     *
     * @param data signal that should be coded by alphabet entropy
     * @return coded signal
     */
    private int[] codeDataECG(double[] data) {
        int N = data.length;

        if (N < AlphabetEntropy.MINIMAL_LENGTH_FOR_EXTRACTION) {
            this.tooSmallArray = true;
            return null;
        }
        int[] alfabetCoded = new int[N - 3];

        for (int j = 0; j < N - 3; j++) {
            if (Math.abs(data[j + 1] - data[j]) <= observationUnchangedTime) {    // first is 0
                if (Math.abs(data[j + 2] - data[j + 1]) <= observationUnchangedTime) { // second is 0
                    if (Math.abs(data[j + 3] - data[j + 2]) <= observationUnchangedTime) { // third is 0
                        alfabetCoded[j] = 0;
                    } else if (data[j + 3] - data[j + 2] > observationUnchangedTime) { // third is +
                        alfabetCoded[j] = 1;
                    } else {    // treci je -
                        alfabetCoded[j] = 2;
                    }
                } else if (data[j + 2] - data[j + 1] > observationUnchangedTime) { // second is +
                    if (Math.abs(data[j + 3] - data[j + 2]) <= observationUnchangedTime) { // third is 0
                        alfabetCoded[j] = 3;
                    } else if (data[j + 3] - data[j + 2] > observationUnchangedTime) { // third is +
                        alfabetCoded[j] = 5;
                    } else {    // treci je -
                        alfabetCoded[j] = 7;
                    }
                } else {    // drugi je -
                    if (Math.abs(data[j + 3] - data[j + 2]) <= observationUnchangedTime) { // third is 0
                        alfabetCoded[j] = 4;
                    } else if (data[j + 3] - data[j + 2] > observationUnchangedTime) { // third is +
                        alfabetCoded[j] = 8;
                    } else {    // treci je -
                        alfabetCoded[j] = 6;
                    }
                }
            } else if (data[j + 1] - data[j] > observationUnchangedTime) { // first is +
                if (Math.abs(data[j + 2] - data[j + 1]) <= observationUnchangedTime) { // second is 0
                    if (Math.abs(data[j + 3] - data[j + 2]) <= observationUnchangedTime) { // third is 0
                        alfabetCoded[j] = 9;
                    } else if (data[j + 3] - data[j + 2] > observationUnchangedTime) { // third is +
                        alfabetCoded[j] = 11;
                    } else {    // treci je -
                        alfabetCoded[j] = 12;
                    }
                } else if (data[j + 2] - data[j + 1] > observationUnchangedTime) { // second is +
                    if (Math.abs(data[j + 3] - data[j + 2]) <= observationUnchangedTime) { // third is 0
                        alfabetCoded[j] = 15;
                    } else if (data[j + 3] - data[j + 2] > observationUnchangedTime) { // third is +
                        alfabetCoded[j] = 19;
                    } else {    // treci je -
                        alfabetCoded[j] = 20;
                    }
                } else {    // drugi je -
                    if (Math.abs(data[j + 3] - data[j + 2]) <= observationUnchangedTime) { // third is 0
                        alfabetCoded[j] = 16;
                    } else if (data[j + 3] - data[j + 2] > observationUnchangedTime) { // third is +
                        alfabetCoded[j] = 21;
                    } else {    // treci je -
                        alfabetCoded[j] = 22;
                    }
                }
            } else { // prvi je -
                if (Math.abs(data[j + 2] - data[j + 1]) <= observationUnchangedTime) { // second is 0
                    if (Math.abs(data[j + 3] - data[j + 2]) <= observationUnchangedTime) { // third is 0
                        alfabetCoded[j] = 10;    // k
                    } else if (data[j + 3] - data[j + 2] > observationUnchangedTime) { // third is +
                        alfabetCoded[j] = 14;    // o
                    } else {    // treci je -
                        alfabetCoded[j] = 13;    // n
                    }
                } else if (data[j + 2] - data[j + 1] > observationUnchangedTime) { // second is +
                    if (Math.abs(data[j + 3] - data[j + 2]) <= observationUnchangedTime) { // third is 0
                        alfabetCoded[j] = 18;    // s
                    } else if (data[j + 3] - data[j + 2] > observationUnchangedTime) { // third is +
                        alfabetCoded[j] = 26;    // aa
                    } else {    // treci je -
                        alfabetCoded[j] = 25;    // z
                    }
                } else {    // drugi je -
                    if (Math.abs(data[j + 3] - data[j + 2]) <= observationUnchangedTime) { // third is 0
                        alfabetCoded[j] = 17;    // r
                    } else if (data[j + 3] - data[j + 2] > observationUnchangedTime) { // third is +
                        alfabetCoded[j] = 24;    // y
                    } else {    // treci je -
                        alfabetCoded[j] = 23;    // x
                    }
                }
            }
        }
        return alfabetCoded;
    }

    /**
     * Private method that counts the appearence individual letters in the coded signal
     *
     * @param codedAlphabet coded signal
     * @return individual letters count
     */
    private int[] countAlphabet(int[] codedAlphabet) {
        int[] alphabetCounted = new int[ALPHABET_SIZE];

        for (int i = 0; i < codedAlphabet.length; i++) {
            alphabetCounted[codedAlphabet[i]]++;
        }

        return alphabetCounted;
    }

    /**
     * Private method that determines whether individual letters appear in the coded signal
     *
     * @param codedAlphabet coded signal
     * @return whether individual letters appear in the coded signal
     */
    private boolean[] existsLetter(int[] codedAlphabet) {
        boolean[] letterExisting = new boolean[ALPHABET_SIZE];
        int i;
        for (i = 0; i < letterExisting.length; i++) {
            letterExisting[i] = false;
        }
        for (i = 0; i < codedAlphabet.length; i++) {
            letterExisting[codedAlphabet[i]] = true;
        }

        return letterExisting;
    }

    /**
     * Calculation of 1D expanded alphabet tessellation entropy.
     * Tessellation entropy is calculated sequentially for miniseries consisting of four measurements
     * mean of entropy for all letters, variance of entropy for all letters, maximum of entropy for all letters,
     * mean of entropy per letter, variance of entropy per letter, maximum of entropy per letter
     * <p>
     * Example:
     * 10.5 11.2 12.1 10.7 11.3 12.0
     * <p>
     * We take first four measurements
     * 10.5 11.2 12.1 10.7
     * <p>
     * The corresponding letter is u (++-)
     * <p>
     * Six points T1 - T6 are determined:
     * T1 = x2 = 11.2, T2 = x3 = 12.1, T3 = x4 = 10.7, T4 = x4+x3-x1 = 11.6, T5 = x4+x3-x1+x4-x2 = 11.1, T6 = T5+x4-x1 = 11.3
     * <p>
     * Six points are next sorted into (A,B,C,D,E,F) and the alphabet entropy is determined as:
     * <p>
     * I = IA + IB + IC + ID + IE + IF = (B-A)/2 * log2((B-A)/2) + (C-A)/2 * log2((C-A)/2) +
     * + (D-B)/2 * log2((D-B)/2) + (E-C)/2 * log2((E-C)/2) + (F-D)/2 * log2((F-D)/2) + (F-E)/2 * log2((F-E)/2)
     * <p>
     * This formula is valid only if all six 6 values are different.
     * In case where some of the measurement values are repeated, some of the members in the formula are used several times, and others are not.
     *
     * @param series        original signal
     * @param codedAlphabet coded signal
     * @return true if the entropy has been calculated, false otherwise
     */
    private boolean calculateExpandedTessellationEntropy(double[] series, int[] codedAlphabet) {
        double[] sixer = new double[6];
        double[] entropies = new double[series.length - 3];
        double temp = 0.0;
        int coded = 0;
        int k = 0;
        int[] ls = new int[ALPHABET_SIZE];
        int countWeird = 0;
        int max = Statistics.maximum(countAlphabet, false);
        if (max == 0) {
            return false;
        }
        averageAlphabetEntropyPerLetter = new double[ALPHABET_SIZE];
        maximumAlphabetEntropyPerLetter = new double[ALPHABET_SIZE];
        varianceAlphabetEntropyPerLetter = new double[ALPHABET_SIZE];
        double[][] alphabetEntropyPerLetter = new double[ALPHABET_SIZE][max];

        for (int i = 0; i < ALPHABET_SIZE; i++) {
            maximumAlphabetEntropyPerLetter[i] = 0.0;
        }

        for (int i = 0; i < ALPHABET_SIZE; i++) {
            for (int j = 0; j < max; j++) {
                alphabetEntropyPerLetter[i][j] = -1.0;
            }
            ls[i] = -1;
        }

        for (int i = 0; i < series.length - 3; i++) {
            if (series[i] >= 5 || series[i + 1] >= 5 || series[i + 2] >= 5 || series[i + 3] >= 5) { // ako je RR interval dulji od 5 sekundi, onda je greška u zapisu
                countWeird++;
                entropies[i] = -1.0;
                continue;
            }
            sixer[0] = series[i] + (series[i + 1] - series[i]);
            sixer[1] = sixer[0] + (series[i + 2] - series[i + 1]);
            sixer[2] = sixer[1] + (series[i + 3] - series[i + 2]);
            sixer[3] = sixer[2] + (series[i + 2] - series[i]);
            sixer[4] = sixer[3] + (series[i + 3] - series[i + 1]);
            sixer[5] = sixer[4] + (series[i + 3] - series[i]);
            Arrays.sort(sixer);
            coded = codedAlphabet[i];
            ls[coded]++;
            if (coded == 0) { // for letter A
                entropies[i] += 0;
                alphabetEntropyPerLetter[coded][ls[coded]] += 0;
                continue;
            }
            int countfirst = 1, countsecond = 1, countthird = 1, countfourth = 1;
            int j;
            for (j = 0; j < 5; j++) {
                if (Math.abs(sixer[j + 1] - sixer[j]) <= observationUnchangedTime) {
                    countfirst++;
                } else break;
            }
            if (countfirst != 1) { // pomak za onoliko mjesta koliko je bilo istih na pocetku
                if (j != 5) {
                    temp = countfirst * (sixer[j + 1] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[j + 1] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                } else { // svi isti
                    entropies[i] += 0; // ustvari suvisno, ali konzistentnosti radi
                    alphabetEntropyPerLetter[coded][ls[coded]] += 0;
                    continue;
                }
                j = j + 1;
            } else { // nema dva ista na pocetku, recimo 56XXXX kombinacija
                temp = (sixer[1] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[1] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                entropies[i] += temp;
                j = 1;
            }
            if (j == 5) { // kombinacija 555556
                temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                entropies[i] += temp;
                continue;
            }
            k = j - 1;
            for (; j < 5; j++) {
                if (Math.abs(sixer[j + 1] - sixer[j]) <= observationUnchangedTime) {
                    countsecond++;
                } else break;
            }
            if (countsecond != 1) {    // j could have also been 2, 3, 4 or 5
                if (j == 2) {
                    temp = countsecond * (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                    for (j = j + 1; j < 5; j++) {
                        if (Math.abs(sixer[j + 1] - sixer[j]) <= observationUnchangedTime) {
                            countthird++;
                        } else break;
                    }
                    if (countthird != 1) {    // j je mogao biti 4 ili 5
                        if (j == 4) {
                            temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                            temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                        } else {
                            temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                        }
                    }
                } else if (j == 3) {
                    temp = countsecond * (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                    if (Math.abs(sixer[5] - sixer[4]) <= observationUnchangedTime) {
                        temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                    } else {
                        temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                        temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                    }
                } else if (j == 4) {
                    if (k != 2) {
                        temp = countsecond * (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    } else {
                        temp = countsecond * (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                    }
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                    temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                } else { // j=5
                    temp = countsecond * (sixer[5] - sixer[k]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[k]) / (2 * sixer[5])) / Math.log10(2.0);
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                }
            } else { // j moze biti 1,2,3,4 ili 5
                if (j == 1) {
                    temp = (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                } else if (j == 2) {
                    temp = (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                } else if (j == 3) {
                    temp = (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                } else if (j == 4) {
                    temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                    alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                    entropies[i] += temp;
                }
                for (j = j + 1; j < 5; j++) {
                    if (Math.abs(sixer[j + 1] - sixer[j]) <= observationUnchangedTime) {
                        countthird++;
                    } else break;
                }
                if (countthird != 1) {    // j je mogao biti 3, 4 ili 5
                    if (j == 4) {
                        temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                        temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                    } else if (j == 5) {
                        temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                    } else { // j==3
                        temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                        if (Math.abs(sixer[5] - sixer[4]) <= observationUnchangedTime) {
                            temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                        } else {
                            temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                            temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                        }
                    }
                } else { // j je mogao biti 2,3,4,5
                    if (j == 2) {
                        temp = (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                    } else if (j == 3) {
                        temp = (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                    } else if (j == 4) {
                        temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                        alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                        entropies[i] += temp;
                    }
                    for (j = j + 1; j < 5; j++) {
                        if (Math.abs(sixer[j + 1] - sixer[j]) <= observationUnchangedTime) {
                            countfourth++;
                        } else break;
                    }
                    if (countfourth != 1) {
                        if (j == 4) {
                            temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                        } else {
                            temp = countfourth * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                        }
                    } else {
                        if (j == 3) {
                            temp = (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                            if (Math.abs(sixer[5] - sixer[4]) <= observationUnchangedTime) {
                                temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                                alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                                entropies[i] += temp;
                            } else {
                                temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                                alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                                entropies[i] += temp;
                                temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                                alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                                entropies[i] += temp;
                            }
                        } else { //if (j==4){
                            temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                            temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);    // moralo je ostati da su zadnja dva broja jednaka
                            alphabetEntropyPerLetter[coded][ls[coded]] += temp;
                            entropies[i] += temp;
                        }
                    }
                }
            }
            entropies[i] = -entropies[i];
        }
        int index = 0;
        double[] tempEntropy = null;
        for (int i = 0; i < ALPHABET_SIZE; i++) {
            if (alphabetEntropyPerLetter[i][0] == -1.0) {
                averageAlphabetEntropyPerLetter[i] = 0.0;
                maximumAlphabetEntropyPerLetter[i] = 0.0;
                varianceAlphabetEntropyPerLetter[i] = 0.0;
                continue;
            }
            for (int j = 0; j < max; j++) {
                if (alphabetEntropyPerLetter[i][j] == -1.0) {
                    index = j;
                    break;
                }
            }
            if (index == 0) {
                index = max;
            }
            tempEntropy = new double[index];
            for (int j = 0; j < index; j++) {
                tempEntropy[j] = -alphabetEntropyPerLetter[i][j];
            }
            averageAlphabetEntropyPerLetter[i] = Statistics.mean(tempEntropy);
            maximumAlphabetEntropyPerLetter[i] = Statistics.maximum(tempEntropy);
            varianceAlphabetEntropyPerLetter[i] = Statistics.variance(tempEntropy, averageAlphabetEntropyPerLetter[i]);
        }
        double[] useableEntropies = new double[entropies.length - countWeird];

        for (int i = 0, j = 0; i < entropies.length; i++) {
            if ((int) (entropies[i]) != -1) {
                useableEntropies[j] = 1.0 + entropies[i];
                j++;
            }
        }

        averageAlphabetEntropy = Statistics.mean(useableEntropies);
        if (averageAlphabetEntropy < 0.0) {
            System.err.print("I am here.");
        }
        varianceAlphabetEntropy = Statistics.variance(useableEntropies, averageAlphabetEntropy);
        if (useableEntropies.length == 0) {
            maximumAlphabetEntropy = 0.0;
        } else {
            maximumAlphabetEntropy = Statistics.maximum(useableEntropies);
        }
        return true;
    }

    /**
     * The method calculates expanded tessellation entropy (alphabet entropy) for a single letter in the alphabet.
     * Note: the method was modified so that equal measurements in the sextuple are considered only those within 1 ms.
     *
     * @param series
     * @param codedAlphabetLetter
     * @return
     */
    public double calculateExpandedTessellationEntropySingleLetter(double[] series, int codedAlphabetLetter) {
        double[] sixer = new double[6];
        double entropy = 0.0;
        double temp = 0.0;
        int k = 0;

        if (series[0] >= 5 || series[1] >= 5 || series[2] >= 5 || series[3] >= 5) { // ako je RR interval dulji od 5 sekundi, onda je greška u zapisu
            return -1.0;
        } else {
            sixer[0] = series[0] + (series[1] - series[0]);
            sixer[1] = sixer[0] + (series[2] - series[1]);
            sixer[2] = sixer[1] + (series[3] - series[2]);
            sixer[3] = sixer[2] + (series[2] - series[0]);
            sixer[4] = sixer[3] + (series[3] - series[1]);
            sixer[5] = sixer[4] + (series[3] - series[0]);
            Arrays.sort(sixer);

            if (codedAlphabetLetter == 0) { // for letter A
                return 0.0;
            } else { // for all other letters
                int countfirst = 1, countsecond = 1, countthird = 1, countfourth = 1;
                int j;
                for (j = 0; j < 5; j++) {
                    if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                        countfirst++;
                    } else break;
                }
                if (countfirst != 1) { // pomak za onoliko mjesta koliko je bilo istih na pocetku
                    if (j != 5) {
                        temp = countfirst * (sixer[j + 1] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[j + 1] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else { // svi isti
                        return 0.0; // ustvari suvisno, ali konzistentnosti radi
                    }
                    j = j + 1;
                } else { // nema dva ista na pocetku, recimo 56XXXX kombinacija
                    temp = (sixer[1] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[1] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                    j = 1;
                }
                if (j == 5) { // kombinacija 555556
                    temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                }
                k = j - 1;
                for (; j < 5; j++) {
                    if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                        countsecond++;
                    } else break;
                }
                if (countsecond != 1) {    // j could have also been 2, 3, 4 or 5
                    if (j == 2) {
                        temp = countsecond * (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                        for (j = j + 1; j < 5; j++) {
                            if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                                countthird++;
                            } else break;
                        }
                        if (countthird != 1) {    // j je mogao biti 4 ili 5
                            if (j == 4) {
                                temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                                temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                            } else {
                                temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                            }
                        }
                    } else if (j == 3) {
                        temp = countsecond * (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                        if (Math.abs(sixer[5] - sixer[4]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                            temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        } else {
                            temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                            temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        }
                    } else if (j == 4) {
                        if (k != 2) {
                            temp = countsecond * (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                        } else {
                            temp = countsecond * (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                        }
                        entropy += temp;
                        temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else { // j=5
                        temp = countsecond * (sixer[5] - sixer[k]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[k]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    }
                } else { // j moze biti 1,2,3,4 ili 5
                    if (j == 1) {
                        temp = (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else if (j == 2) {
                        temp = (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else if (j == 3) {
                        temp = (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else if (j == 4) {
                        temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    }
                    for (j = j + 1; j < 5; j++) {
                        if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                            countthird++;
                        } else break;
                    }
                    if (countthird != 1) {    // j je mogao biti 3, 4 ili 5
                        if (j == 4) {
                            temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                            temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        } else if (j == 5) {
                            temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        } else { // j==3
                            temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                            if (Math.abs(sixer[5] - sixer[4]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                                temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                            } else {
                                temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                                temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                            }
                        }
                    } else { // j je mogao biti 2,3,4,5
                        if (j == 2) {
                            temp = (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        } else if (j == 3) {
                            temp = (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        } else if (j == 4) {
                            temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        }
                        for (j = j + 1; j < 5; j++) {
                            if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                                countfourth++;
                            } else break;
                        }
                        if (countfourth != 1) {
                            if (j == 4) {
                                temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                            } else {
                                temp = countfourth * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                            }
                        } else {
                            if (j == 3) {
                                temp = (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                                if (Math.abs(sixer[5] - sixer[4]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                                    temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                                    entropy += temp;
                                } else {
                                    temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                                    entropy += temp;
                                    temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                                    entropy += temp;
                                }
                            } else { //if (j==4){
                                temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                                temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);    // moralo je ostati da su zadnja dva broja jednaka
                                entropy += temp;
                            }
                        }
                    }
                }
            }
            entropy = -entropy + 1.0;
        }
        return entropy;
    }

    /**
     * The method calculates expanded tessellation entropy (alphabet entropy) for a single letter in the alphabet.
     * Note: the method was modified so that equal measurements in the sextuple are considered only those within 1 ms.
     *
     * @param series
     * @param codedAlphabetLetter
     * @return
     */
    public double calculateExpandedTessellationEntropyEnhancedSingleLetter(double[] series, int codedAlphabetLetter) {
        double[] sixer = new double[15];
        double entropy = 0.0;
        double temp = 0.0;
        int k = 0;

        if (series[0] >= 5 || series[1] >= 5 || series[2] >= 5 || series[3] >= 5 || series[4] >= 5 || series[5] >= 5) { // ako je RR interval dulji od 5 sekundi, onda je greška u zapisu
            return -1.0;
        } else {
            sixer[0] = series[0] + (series[1] - series[0]);
            sixer[1] = sixer[0] + (series[2] - series[1]);
            sixer[2] = sixer[1] + (series[3] - series[2]);
            sixer[3] = sixer[2] + (series[4] - series[3]);
            sixer[4] = sixer[3] + (series[5] - series[4]);
            sixer[5] = sixer[4] + (series[2] - series[0]);
            sixer[6] = sixer[5] + (series[3] - series[1]);
            sixer[7] = sixer[6] + (series[4] - series[2]);
            sixer[8] = sixer[7] + (series[5] - series[3]);
            sixer[9] = sixer[8] + (series[3] - series[0]);
            sixer[10] = sixer[9] + (series[4] - series[1]);
            sixer[11] = sixer[10] + (series[5] - series[2]);
            sixer[12] = sixer[11] + (series[4] - series[0]);
            sixer[13] = sixer[12] + (series[5] - series[1]);
            sixer[14] = sixer[13] + (series[5] - series[0]);

            Arrays.sort(sixer);

            if (codedAlphabetLetter == 0) { // for letter A
                return 0.0;
            } else {
                int j;
                int countFirst = 1;
                int countSame = 1;
                for (j = 0; j < 14; j++) { // slučaj prvoga koji se ponavlja - treba ga se uzeti toliko puta koliko se ponavlja
                    if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                        countFirst++;
                    } else break;
                }
                if (countFirst != 1) { // npr. AAABCD...K
                    entropy += countFirst * (sixer[0] + sixer[1]) / (2 * sixer[14]) * Math.log10((sixer[0] + sixer[1]) / (2 * sixer[14])) / Math.log10(2.0);
                } else {
                    entropy += (sixer[0] + sixer[1]) / (2 * sixer[14]) * Math.log10((sixer[0] + sixer[1]) / (2 * sixer[14])) / Math.log10(2.0);
                }
                for (j = countFirst; j < 14; j++) {
                    if (Math.abs(sixer[j + 1] - sixer[j - 1]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                        countSame++; // sta se dogadja ako ima vise istih na kraju?
                    } else {
                        if (countSame != 1) { // npr. AAABCDDDEFG  (u sredini mora biti najmanje tri ista da bi bio problem, dva nisu problem
                            entropy += (sixer[j - 1] - sixer[j - 3]) / (2 * sixer[5]) * Math.log10((sixer[1] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                        }
                    }
                }
            }


            //else { // for all other letters
            int countfirst = 1, countsecond = 1, countthird = 1, countfourth = 1;
            int j;
            for (j = 0; j < 5; j++) {
                if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                    countfirst++;
                } else break;
            }
            if (countfirst != 1) { // pomak za onoliko mjesta koliko je bilo istih na pocetku
                if (j != 5) {
                    temp = countfirst * (sixer[j + 1] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[j + 1] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                } else { // svi isti
                    return 0.0; // ustvari suvisno, ali konzistentnosti radi
                }
                j = j + 1;
            } else { // nema dva ista na pocetku, recimo 56XXXX kombinacija
                temp = (sixer[1] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[1] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                entropy += temp;
                j = 1;
            }
            if (j == 5) { // kombinacija 555556
                temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                entropy += temp;
            }
            k = j - 1;
            for (; j < 5; j++) {
                if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                    countsecond++;
                } else break;
            }
            if (countsecond != 1) {    // j could have also been 2, 3, 4 or 5
                if (j == 2) {
                    temp = countsecond * (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                    for (j = j + 1; j < 5; j++) {
                        if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                            countthird++;
                        } else break;
                    }
                    if (countthird != 1) {    // j je mogao biti 4 ili 5
                        if (j == 4) {
                            temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                            temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        } else {
                            temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        }
                    }
                } else if (j == 3) {
                    temp = countsecond * (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                    if (Math.abs(sixer[5] - sixer[4]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                        temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else {
                        temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                        temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    }
                } else if (j == 4) {
                    if (k != 2) {
                        temp = countsecond * (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    } else {
                        temp = countsecond * (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                    }
                    entropy += temp;
                    temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                } else { // j=5
                    temp = countsecond * (sixer[5] - sixer[k]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[k]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                }
            } else { // j moze biti 1,2,3,4 ili 5
                if (j == 1) {
                    temp = (sixer[2] - sixer[0]) / (2 * sixer[5]) * Math.log10((sixer[2] - sixer[0]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                } else if (j == 2) {
                    temp = (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                } else if (j == 3) {
                    temp = (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                } else if (j == 4) {
                    temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                    entropy += temp;
                }
                for (j = j + 1; j < 5; j++) {
                    if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                        countthird++;
                    } else break;
                }
                if (countthird != 1) {    // j je mogao biti 3, 4 ili 5
                    if (j == 4) {
                        temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                        temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else if (j == 5) {
                        temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else { // j==3
                        temp = countthird * (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                        if (Math.abs(sixer[5] - sixer[4]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                            temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        } else {
                            temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                            temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        }
                    }
                } else { // j je mogao biti 2,3,4,5
                    if (j == 2) {
                        temp = (sixer[3] - sixer[1]) / (2 * sixer[5]) * Math.log10((sixer[3] - sixer[1]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else if (j == 3) {
                        temp = (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    } else if (j == 4) {
                        temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                        entropy += temp;
                    }
                    for (j = j + 1; j < 5; j++) {
                        if (Math.abs(sixer[j + 1] - sixer[j]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                            countfourth++;
                        } else break;
                    }
                    if (countfourth != 1) {
                        if (j == 4) {
                            temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        } else {
                            temp = countfourth * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                        }
                    } else {
                        if (j == 3) {
                            temp = (sixer[4] - sixer[2]) / (2 * sixer[5]) * Math.log10((sixer[4] - sixer[2]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                            if (Math.abs(sixer[5] - sixer[4]) <= NO_CHANGE_THRESHOLD_DEFAULT) {
                                temp = 2 * (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                            } else {
                                temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                                temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);
                                entropy += temp;
                            }
                        } else { //if (j==4){
                            temp = (sixer[5] - sixer[3]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[3]) / (2 * sixer[5])) / Math.log10(2.0);
                            entropy += temp;
                            temp = (sixer[5] - sixer[4]) / (2 * sixer[5]) * Math.log10((sixer[5] - sixer[4]) / (2 * sixer[5])) / Math.log10(2.0);    // moralo je ostati da su zadnja dva broja jednaka
                            entropy += temp;
                        }
                    }
                }
            }
        }
        entropy = -entropy + 1.0;
        //}
        return entropy;
    }
}