package multisab.processing.commonSignalFeatures.nonlinear.entropy;

import multisab.processing.commonSignalFeatures.timeDomain.statisticMeasure.DistanceMeasure;
import multisab.processing.commonSignalFeatures.timeDomain.statisticMeasure.Statistics;

import java.util.List;

public class FuzzyApEn {
    public static final int MINIMAL_LENGTH_FOR_EXTRACTION = 15;

    /**
     * The private method calculates the number phi(m) using exponential (Gaussian) member function
     *
     * @param series signal given as a field of type double
     * @param m      m factor used in calculating phi(m)
     * @param r      radius for data inclusion
     * @return phi(m)
     */
    private static double calculateFuzzyApEnSingleStepOfM(double[] series, int m, double r) {
        int size = series.length;

        // Step 1 of FuzzyApEn
        double[][] vectors = new double[size - m][m];
        double baseline = 0.0;

        for (int i = 0; i < vectors.length; i++) {
            baseline = 0.0;
            for (int j = 0; j < m; j++) {
                baseline += series[i + j];
            }
            baseline /= m;
            for (int j = 0; j < m; j++) {
                vectors[i][j] = series[i + j] - baseline;
            }
        }

        // Steps 2 and 3 of FuzzyApEn
        double[] membershipFunction = new double[vectors.length];
        double[] numberCMIR = new double[vectors.length];
        for (int i = 0; i < vectors.length; i++) {
            for (int j = 0; j < vectors.length; j++) {
                if (i == j) continue;
                membershipFunction[i] += Math.exp(-Math.pow(DistanceMeasure.maximumAbsoluteDifference(vectors[i], vectors[j]), 2.0) / r);
            }
            numberCMIR[i] = (double) membershipFunction[i] / vectors.length;
        }

        // Step 4 of FuzzyApEn
        double numberPHI = 0.0;
        for (int i = 0; i < vectors.length; i++) {
            numberPHI += Math.log(numberCMIR[i]);
        }
        numberPHI /= vectors.length;

        return numberPHI;
    }

    /**
     * Calculates fuzzy approximate entropy for the time series given in the field of type double
     *
     * @param series signal given as a field of type double
     * @param m      m factor used in calculating phi(m)
     * @param r      radius for data inclusion
     * @return FuzzyApEn
     */
    public static double calculateFuzzyApEn(double[] series, int m, double r) {
        double fapen;

        double phi0 = FuzzyApEn.calculateFuzzyApEnSingleStepOfM(series, m, r);
        double phi1 = FuzzyApEn.calculateFuzzyApEnSingleStepOfM(series, m + 1, r);

        fapen = phi0 - phi1;

        return fapen;
    }

    /**
     * Calculates approximate entropy for the time series given in the list of type Double
     *
     * @param series signal given as a field of type double
     * @param m      m factor used in calculating phi(m)
     * @param r      radius for data inclusion
     * @return ApEn
     */
    public static double calculateFuzzyApEn(List<Double> series, int m, double r) {
        double apen;
        double[] serie = new double[series.size()];
        for (int i = 0; i < series.size(); i++) {
            serie[i] = series.get(i).doubleValue();
        }

        double phi0 = FuzzyApEn.calculateFuzzyApEnSingleStepOfM(serie, m, r);
        double phi1 = FuzzyApEn.calculateFuzzyApEnSingleStepOfM(serie, m + 1, r);

        apen = phi0 - phi1;

        return apen;
    }

    /**
     * Calculates the maximum of FuzzyApEn dependent of the radius r for the time series given in the field of type double
     *
     * @param series signal given as a field of type double
     * @param m      m factor used in calculating phi(m)
     * @return a pair {maximum of FuzzyApEn, radius r for the maximum of ApEn}
     */
    public static double[] calculateMaxFuzzyApEn(double[] series, int m) {
        double apenMax = 0.0;
        double maxIndex = 0.0;
        double stdev = Statistics.standardDeviation(series);
        double phi0, phi1, apen;
        for (int i = 1; i <= 50; i++) {
            phi0 = FuzzyApEn.calculateFuzzyApEnSingleStepOfM(series, m, 0.01 * i * stdev);
            phi1 = FuzzyApEn.calculateFuzzyApEnSingleStepOfM(series, m + 1, 0.01 * i * stdev);
            apen = phi0 - phi1;
            if (apen > apenMax) {
                apenMax = apen;
                maxIndex = 0.01 * i;
            }
        }
        return new double[]{apenMax, maxIndex};
    }
}
