package multisab.processing.commonSignalFeatures.timeFrequencyDomain;

import multisab.processing.commonSignalFeatures.timeDomain.statisticMeasure.Statistics;

/**
 * The class calculated standard deviation of haar wavelet transform as proposed by Teich et al.
 * "Heart rate variability, measures and models", 2001.
 * <p>
 * General formula:
 * <p>
 * HWSTDEV(m) = sqrt( 1/N-1*(sum(i=1,i<=N)(W(m,i)-average(W(m)))^2)),
 * <p>
 * where W(m)= 1/sqrt(m)*sum(j=0,j<=L-1)(RR[j]*H((j/m)-n)),
 * <p>
 * H(x) = {+1, 0<x<m/2
 * -1, m/2<x<m
 * 0, otherwise
 * <p>
 * We simplify the method for Haar wavelet as proposed by Teich to calculate W(m,i):
 * <p>
 * W(m,i) = 1/sqrt(m)*(sum(i,i=i+m/2)(RR[i])-sum(i=i+m/2;i<m)(RR[i]))
 * <p>
 * Wavelet transform of scale m=32 and a large number of RR intervals (32000+) has been shown to be a perfect indicator for normal / CHF patient distinction
 * A special case where scale=2 yields RMSSD
 *
 * @author Alan Jovic
 */
public class HaarWaveletStandardDeviation {
    public static final int DEFAULT_HAAR_WAVELET_SCALE = 8;
    public static final double calculateHaarWaveletSTDEV(double[] segment, int scale) throws Exception {
        int intervals = segment.length / scale;
        double[] waves = new double[intervals];

        if (intervals <= 2) {
            System.err.println("Too few wavelet intervals!");
            return 0.0;
        }
        if (scale < 2) {
            System.err.println("Too small scale!");
            return 0.0;
        }
        if (scale % 2 == 0) {
            for (int i = 0; i < intervals; i++) {
                for (int j = 0; j < scale / 2; j++) {
                    waves[i] += segment[i * scale + j];
                }
                for (int j = scale / 2; j < scale; j++) {
                    waves[i] -= segment[i * scale + j];
                }
            }
        } else {
            for (int i = 0; i < intervals; i++) {
                for (int j = 0; j < scale / 2; j++) {
                    waves[i] += segment[i * scale + j];
                }
                for (int j = 1 + scale / 2; j < scale; j++) {
                    waves[i] -= segment[i * scale + j];
                }
            }
        }
        return Statistics.standardDeviation(waves);
    }
}