package multisab.processing.hrvAnalysis;

import com.aparapi.Kernel;
import multisab.processing.commonSignalFeatures.timeDomain.statisticMeasure.Statistics;

import java.util.Random;

public class StandardDeviationRatio {
    public static final int MINIMAL_LENGTH_FOR_EXTRACTION = 5;
    /*
     * - cited from "Mining Physiological Conditions from Heart Rate Variability Analysis",
     * Che-Wei Lin, Jeen-Shing Wang and Pau-Choo Chung, IEEE Computational Intelligence Magazine, 5(1):50-58, Feb 2010
     *
     *
     * original article:
     *
     * J Auton Nerv Syst. 1997 Jan 12;62(1-2):79-84.
        A new method of assessing cardiac autonomic function and its comparison with spectral multisab.processing.analysis and coefficient of variation of R-R interval.
        Toichi M, Sugiura T, Murai T, Sengoku A.
     */
    //PROVJERITI JOŠ
    private double SD1, SD2;

    public StandardDeviationRatio() {

    }

    /**
     * Centar elipse određen je prosječnom duljinom segmenta
     * SD1 označava standardnu devijaciju udaljenosti tocaka od osi y = x,
     * a SD2 je standardna devijacija tocaka od osi y = -x + mean,
     * gdje je mean prosjecna duljina segmenta. SD1 devijacija odredjuje sirinu elipse, dok SD2 odredjuje duljinu elipse.
     *
     * @param segment
     */
    public static double calculateStandardDeviationRatio(double[] segment) {
        double mean = Statistics.mean(segment);
        double[] distancesFromYequalsX = new double[segment.length - 1];
        double[] distancesFromYequalsMinusXplusMean = new double[segment.length - 1];
        double[] rez = new double[3];
        double root2 = Math.sqrt(2);

        for (int i = 0; i < segment.length - 1; i++) {
            distancesFromYequalsX[i] = Math.abs(segment[i] - segment[i + 1]) * root2 / 2.0; // udaljenost tocke (xi,yi) od y=x osi: d1 = |xi - yi|/sqrt(2)
            //Ova ispod formula se ne slaže s onom koju sam našla
            distancesFromYequalsMinusXplusMean[i] = Math.abs(segment[i] + segment[i + 1] - 2 * mean) / root2; // udaljenost tocke (xi,yi) od y = -x + RRm osi: d1 = |xi + yi - RRm|/sqrt(2)
        }
        rez[0] = Statistics.standardDeviation(distancesFromYequalsX);
        rez[1] = Statistics.standardDeviation(distancesFromYequalsMinusXplusMean);
        rez[2] = rez[0] / rez[1];
        return rez[2];
    }


    /**
     * Paralelna verzija izracuna omjera standardnih devijacija za HRV
     *
     * @param segment
     * @return
     */
    public static double[] standardDeviationRatioP(double[] segment) {
        double mean = Statistics.mean(segment);
        final double[] distancesFromYequalsX = new double[segment.length - 1];
        final double[] distancesFromYequalsMinusXplusMean = new double[segment.length - 1];
        double[] rez = new double[3];
        double root2 = Math.sqrt(2);

        Kernel kernel = new Kernel() {
            @Override
            public void run() {
                int i = getGlobalId();
                distancesFromYequalsX[i] = abs(segment[i] - segment[i + 1]) * root2 / 2.0; // udaljenost tocke (xi,yi) od y=x osi: d1 = |xi - yi|/sqrt(2)
                distancesFromYequalsMinusXplusMean[i] = abs(segment[i] + segment[i + 1] - 2 * mean) / root2; // udaljenost tocke (xi,yi) od y = -x + RRm osi: d1 = |xi + yi - RRm|/sqrt(2)
            }
        };
        kernel.setExecutionMode(Kernel.EXECUTION_MODE.GPU);
        kernel.execute(segment.length - 1);
        rez[0] = Statistics.standardDeviation(distancesFromYequalsX);
        rez[1] = Statistics.standardDeviation(distancesFromYequalsMinusXplusMean);
        rez[2] = rez[0] / rez[1];
        return rez;
    }
    /*
    public static void main(String[] args) {
        double[] rez1, rez2;
        long time1 = System.currentTimeMillis();
        java.util.Random r = new Random();
        double[] series = new double[20000000];
        for (int i = 0; i < series.length; i++) {
            series[i] = r.nextInt(10);
        }
        long time2 = System.currentTimeMillis();
        rez1 = StandardDeviationRatio.standardDeviationRatio(series);
        long time3 = System.currentTimeMillis();
        rez2 = StandardDeviationRatio.standardDeviationRatioP(series);
        long time4 = System.currentTimeMillis();
        System.err.print("Serial: rez = " + rez1[2] + " time elapsed: " + (time3 - time2) + "; Paralel: rez = " + rez2[2] + " time elapsed: " + (time4 - time3));

    }*/

    /**
     * Cardiac sympathetic index = SD2/SD1
     *
     * @return SD2/SD1
     */
    public double getCSI() {
        return SD2 / SD1;
    }

    /**
     * Cardiac vagal index = log10(16*SD1*SD2)
     *
     * @return log10(16*SD1*SD2)
     */
    public double getCVI() {
        return Math.log10(16 * SD1 * SD2);
    }

    public double getSD1() {
        return this.SD1;
    }

    public double getSD2() {
        return this.SD2;
    }

    public double getSD1SD2Ratio() {
        return SD1 / SD2;
    }
}
