package multisab.processing.hrvAnalysis;

import multisab.processing.commonSignalFeatures.timeDomain.statisticMeasure.Statistics;

public class TimeDomainAnalysisHRV {
    public static final int MINIMAL_LENGTH_FOR_EXTRACTION = 10;
    private static final double BIN_SIZE = 0.0078125;

    /**
     * Calculates the mean NN interval length
     *
     * @param rrIntervalsSeries
     * @return
     */
    public static final double calculateAVNN(double[] rrIntervalsSeries) {
        return Statistics.mean(rrIntervalsSeries);
    }

    /**
     * Calculates the inverse of mean NN interval length
     *
     * @param rrIntervalsSeries
     * @return
     */
    public static final double calculate1_AVNN(double[] rrIntervalsSeries) {
        return 1.0/TimeDomainAnalysisHRV.calculateAVNN(rrIntervalsSeries);
    }
    /**
     * Standard deviation of all NN intervals
     *
     * @param rrIntervalsSeries
     * @return
     */
    public static final double calculateSDNN(double[] rrIntervalsSeries) {
        return Statistics.standardDeviation(rrIntervalsSeries);
    }

    /**
     * The square root of the mean of the sum of the squares of differences between adjacent NN intervals
     *
     * @param rrIntervalsSeries
     * @return
     */
    public static final double calculateRMSSD(double[] rrIntervalsSeries) {
        if (rrIntervalsSeries.length < 2){
            System.err.println("Too short time segment for RMSSD calculation");
            return 0.0;
        }
        double[] differences = new double[rrIntervalsSeries.length - 1];

        for (int i = 0; i < rrIntervalsSeries.length - 1; i++) {
            differences[i] = rrIntervalsSeries[i + 1] - rrIntervalsSeries[i];
        }
        double rmssd = Statistics.rootMeanSquare(differences);

        return rmssd;
    }

    /**
     * Standard deviation of differences between adjacent NN intervals
     *
     * @param rrIntervalsSeries
     * @return
     */
    public static final double calculateSDSD(double[] rrIntervalsSeries) {
        if (rrIntervalsSeries.length < 2) {
            System.err.println("Too short time segment for SDSD calculation");
            return 0.0;
        }
        double[] differences = new double[rrIntervalsSeries.length - 1];

        for (int i = 0; i < rrIntervalsSeries.length - 1; i++) {
            differences[i] = rrIntervalsSeries[i + 1] - rrIntervalsSeries[i];
        }
        double sdsd = Statistics.standardDeviation(differences);

        return sdsd;
    }

    /**
     * NNX count divided by the total number of all NN intervals
     *
     * @param rrIntervalsSeries HRV RR series
     * @param X                 double (percentage) - usually 50 (ms)
     * @return
     */
    public static final double calculatePNNX(double[] rrIntervalsSeries, int X) {
        int count = 0;
        double factor = (double) (X) / 1000;
        for (int i = 1; i < rrIntervalsSeries.length; i++) {
            if (Math.abs(rrIntervalsSeries[i] - rrIntervalsSeries[i - 1]) >= factor) {
                count++;
            }
        }
        return (double) count * 100 / rrIntervalsSeries.length;
    }

    /**
     * Number of pairs of adjacent NN intervals differing by more than X ms in the entire recording
     *
     * @param rrIntervalsSeries HRV RR series
     * @param X                 double (percentage) - usually 50 (ms)
     * @return
     */
    public static final int calculateNNXCount(double[] rrIntervalsSeries, int X) {
        int count = 0;
        double factor = (double) (X) / 1000;
        for (int i = 1; i < rrIntervalsSeries.length; i++) {
            if (Math.abs(rrIntervalsSeries[i] - rrIntervalsSeries[i - 1]) >= factor) {
                count++;
            }
        }
        return count;
    }

    /**
     * Standard deviation of the averages of NN intervals in all 5-min segments of the entire recording
     *
     * @param rrIntervalsSeries
     * @param numberOfSegments
     * @return
     */
    public static final double calculateSDANN(double[] rrIntervalsSeries, int numberOfSegments) {
        int length = (int) ((double) (rrIntervalsSeries.length) / numberOfSegments);
        if (length == 0){
            System.err.println("Too short time segment for SDANN calculation");
            return 0.0;
        }
        double mean = Statistics.mean(rrIntervalsSeries);
        double tempSum = 0.0;
        double sum = 0.0;
        int d = 0;

        for (int i = 0; i < rrIntervalsSeries.length; i++) {
            if (d != length) {
                tempSum += rrIntervalsSeries[i];
                d++;
            } else {
                tempSum /= length;
                sum += (tempSum - mean) * (tempSum - mean);
                d = 0;
            }
        }
        sum /= numberOfSegments - 1;
        sum = Math.sqrt(sum);

        return sum;
    }

    /**
     * Mean of the standard deviations of all NN intervals for all 5-min segments of the entire recording
     *
     * @param rrIntervalsSeries
     * @param numberOfSegments
     * @return
     */
    public static final double calculateSDNNindex(double[] rrIntervalsSeries, int numberOfSegments) {
        int length = (int) ((double) (rrIntervalsSeries.length) / numberOfSegments);
        if (length == 0){
            System.err.println("Too short time segment for SDNNindex calculation");
            return 0.0;
        }
        double mean = 0.0;
        int d = 0;
        double[] fiveMinSegment = new double[length];

        for (int i = 0; i < rrIntervalsSeries.length; i++) {
            if (d != length) {
                fiveMinSegment[d] = rrIntervalsSeries[i];
                d++;
            } else {
                mean += calculateSDNN(fiveMinSegment);
                d = 0;
            }
        }
        mean /= numberOfSegments;
        return mean;
    }

    /**
     * Total number of all NN intervals divided by the height of the histogram of all NNintervals measured
     * on a discrete scale with bins of 1/128 s
     * <p>
     * (total number of NN intervals)/(number of NN intervals in the modal bin)
     * //to determine the size of a bin, we use the smallest difference larger than 0 between the lengths of NN intervals
     * the size of the bin is 1/128 Hz^-1 = 7.8125 ms
     *
     * @param rrIntervalsSeries
     * @return
     */
    public static final double calculateHRVTriangularIndex(double[] rrIntervalsSeries) {
        if (rrIntervalsSeries.length < 2){
            System.err.println("Too short time segment for HRV_TRIANGULAR_INDEX calculation");
            return 0.0;
        }
        double min = Statistics.minimum(rrIntervalsSeries, 0, rrIntervalsSeries.length, false);
        double max = Statistics.maximum(rrIntervalsSeries);

        double noBins = (max - min) / BIN_SIZE;
        int[] bins = new int[(int) noBins + 1];

        double distance;
        double lowerLim;


        for (int i = 0; i < rrIntervalsSeries.length; i++) {
            distance = rrIntervalsSeries[i] - min;
            lowerLim = distance / BIN_SIZE;
            bins[(int) lowerLim] += 1;
        }
        double rez = (double) (rrIntervalsSeries.length) / Statistics.maximum(bins, false);
        return rez;
    }

    /**
     * Baseline width of the minimum square difference triangular interpolation of the highest peak of the
     * histogram of all NN intervals
     *
     * @param rrIntervalsSeries
     * @return
     */
    public static final double calculateTINN(double[] rrIntervalsSeries) {
        if (rrIntervalsSeries.length < 2){
            System.err.println("Too short time segment for TINN calculation");
            return 0.0;
        }
        double min = Statistics.minimum(rrIntervalsSeries, 0, rrIntervalsSeries.length, false);
        double max = Statistics.maximum(rrIntervalsSeries);

        double noBins = (max - min) / BIN_SIZE;
        int[] bins = new int[(int) noBins + 1];

        double distance;
        double lowerLim;


        for (int i = 0; i < rrIntervalsSeries.length; i++) {
            distance = rrIntervalsSeries[i] - min;
            lowerLim = distance / BIN_SIZE;
            bins[(int) lowerLim] += 1;
        }
        int maxBins = Statistics.maximum(bins, false);
        int topIndex = 0;
        int M = bins.length - 1, N = 0;
        for (int i = 0; i < bins.length; i++) {
            if (bins[i] == maxBins) {
                topIndex = i;
                break;
            }
        }
        for (int i = topIndex - 1; i >= 0; i--) {
            if (bins[i] == 0) {
                N = i;
                break;
            }
        }
        for (int i = topIndex + 1; i < bins.length; i++) {
            if (bins[i] == 0) {
                M = i;
                break;
            }
        }
        return M - N;
    }
}
