package multisab.processing.preprocessing.morphologicalOperations;

import multisab.processing.preprocessing.otherFunctions.ArrayFunctions;
import multisab.processing.preprocessing.otherFunctions.FindLocalExtremes;

/**
 * @Author: Friganovic
 *
 *
 * Mathematical morphology operators
 * Includes:
 *  Basic operators: erosion,dilation
 *  Derived operators: opening, closing
 *  ECG filtering: MMF
 *
 * References:
 * Sun Y, Chan KL, Krishnan SM (2002) ECG signal conditioning by morphological filtering. Comput Biol Med 32:465–479
 *
 *
 */
public class MMOperators {

    public static double[] erosion(double[] f, double[] B) {

        int N = f.length;
        int M = B.length;
        double[] pom = new double[2]; //help variable for finding max and min

        double[] y = new double[N];
        double[] temp = new double[M];

        for (int n = 0; n < N; n++) {

            if (n < Math.round((M-1)/2)) {
                for (int m = 0; m < Math.round(M/2); m++) {
                    temp[m] = f[n + m] - B[m + Math.round((M-1)/2)];
                }

                pom = FindLocalExtremes.findMinimum(temp,0,Math.round((M/2)));
                y[n] = pom[0];
            }
            else if (n >= N - Math.round((M+1)/2)) {
                for (int m = 0; m < Math.round(M/2); m++){
                    temp[m] = f[n] - B[m];
                }
                pom = FindLocalExtremes.findMinimum(temp,0,Math.round((M+1)/2));
                y[n] = pom[0];
            }
            else {
                for (int m = 0; m < M; m++){
                    temp[m] = f[n - Math.round((M-1)/2) + m] - B[m];
                }
                pom = FindLocalExtremes.findMinimum(temp,0,M);
                y[n] = pom[0];
            }
        }

        return y;
    }
    public static double[] dilation(double[] f, double[] B) {

        int N = f.length;
        int M = B.length;
        double[] pom = new double[2]; //help variable for finding max and min

        double[] y = new double[N];
        double[] temp = new double[M];

        for (int n = 0; n < N; n++) {

            if (n < Math.round((M-1)/2)) {
                for (int m = 0; m < Math.round(M/2); m++) {
                    temp[m] = f[n + m] + B[m + Math.round((M-1)/2)];
                }

                pom = FindLocalExtremes.findMaximum(temp,0,Math.round(M/2));
                y[n] = pom[0];
            }
            else if (n >= N - Math.round((M+1)/2)) {
                for (int m = 0; m < Math.round(M/2); m++){
                    temp[m] = f[n] + B[m];
                }
                pom = FindLocalExtremes.findMaximum(temp,0,Math.round((M+1)/2));
                y[n] = pom[0];
            }
            else {
                for (int m = 0; m < M; m++){
                    temp[m] = f[n - Math.round((M - 1)/2) + m] + B[m];
                }
                pom = FindLocalExtremes.findMaximum(temp,0,M);
                y[n] = pom[0];
            }
        }

        return y;
    }
    public static double[] opening(double[] f, double[] B){

        double [] temp = dilation(f,B);
        double [] y = erosion(temp,B);

        return y;
    }
    public static double[] closing(double[] f, double[] B){

        double [] temp = erosion(f,B);
        double [] y = dilation(temp,B);

        return y;
    }
    public static double[] MMF(double[] fo, double fs){
        //Baseline correction

        //Parameter initialization
        int Lo = (int)Math.round(0.2*fs); //length of structuring element Bo
        int Lc = (int)Math.round(1.5*Lo); //length of structuring element Bc

        double [] Bo = new double[Lo];
        double [] Bc = new double[Lc];

        int N = fo.length;

        //MF baseline correction
        double [] fb = closing(opening(fo,Bo),Bc);
        double [] fbc = new double[N];
        for (int i = 0; i < N; i++){
            fbc[i] = fo[i] - fb[i];
        }

        //Noise suppresion

        //Parameter initialization
        double [] B1 = {0,1,5,1,0}; //empirical
        double [] B2 = {0,0,0,0,0}; //from article

        double [] mmf1 = erosion(dilation(fbc,B1),B2);
        double [] mmf2 = dilation(erosion(fbc,B2),B1);

        double [] mmf = new double[mmf1.length];
        for (int i = 0; i < mmf.length; i++){
            mmf[i] = 0.5 * (mmf1[i] + mmf2[i]);
        }

    return mmf;
    }
    public static double[] MMF(double[] fo, double fs, double[] B1, double[] B2){
        //Baseline correction

        //Parameter initialization
        int Lo = (int)Math.round(0.2*fs); //length of structuring element Bo
        int Lc = (int)Math.round(1.5*Lo); //length of structuring element Bc

        double [] Bo = new double[Lo];
        double [] Bc = new double[Lc];

        int N = fo.length;

        //MF baseline correction
        double [] fb = closing(opening(fo,Bo),Bc);
        double [] fbc = new double[N];
        for (int i = 0; i < N; i++){
            fbc[i] = fo[i] - fb[i];
        }

        //Noise suppresion

        //Parameter initialization

        double [] mmf1 = erosion(dilation(fbc,B1),B2);
        double [] mmf2 = dilation(erosion(fbc,B2),B1);

        double [] mmf = new double[mmf1.length];
        for (int i = 0; i < mmf.length; i++){
            mmf[i] = 0.5 * (mmf1[i] + mmf2[i]);
        }

        return mmf;
    }


    public static double[] MMD(double[] ecg, double fs){

        int s = (int)Math.floor(0.06*fs);
        double[] mdf = new double[ecg.length];

        double[] temp1;
        double[] temp2;



        for (int i = s; i < ecg.length - s; i++){
            temp1 = FindLocalExtremes.findMaximum(ecg,i-s,i+s);
            temp2 =  FindLocalExtremes.findMinimum(ecg,i-s,i+s);

            mdf[i] = (temp1[0] + temp2[0] - 2*ecg[i])/s;
        }
        return mdf;
    }



}
