#include "CentralForceOptimization.h"
#include <iostream>



class CentralForceOptimization::Probe
{
public:
	std::vector<double> &position;		// trenutna pozicija
	std::vector<double> position_prev;	// pozicija u prethodnom koraku
	std::vector<double> acceleration;	// vektor akceleracije

	Probe(std::vector<double> &pos) : position(pos) {
		position_prev = position;
		for (unsigned int i = 0; i < pos.size(); i++) {
			acceleration.push_back(0);
		}
	}
};



CentralForceOptimization::CentralForceOptimization() {
	name_ = "CentralForceOptimization";	
}


bool CentralForceOptimization::initialize(StateP state) {
	// pocetni globalni maksimum
	global_best = pow(10, 4200);

	// provjera je li genotip tipa FloatingPoint
	FloatingPointP flp (new FloatingPoint::FloatingPoint);
	if(state->getGenotypes()[0]->getName() != flp->getName()) {
		ECF_LOG_ERROR(state, "Error: CFO algorithm only accepts a FloatingPoint genotype!");
		throw ("");
	}

	// ucitavanje broja dimenzija funkcije
	voidP Dimension = state->getGenotypes()[0]->getParameterValue(state, "dimension");
	dimensions = *((uint*) Dimension.get());

	// pocetne granice prostora
	voidP lBound = state->getGenotypes()[0]->getParameterValue(state, "lbound");
	xmin_init = *((double*)lBound.get());
	voidP uBound = state->getGenotypes()[0]->getParameterValue(state, "ubound");
	xmax_init = *((double*)uBound.get());

	// parametri faktora vracanja
	Finit = 0.5;
	Fdelta = 0.1;
	Fmin = 0.05;

	// maksimalni broj letjelica po dimenziji
	if (dimensions <= 6) max_pop_per_dimension = 14;
	else if (dimensions <= 11) max_pop_per_dimension = 10;
	else if (dimensions <= 20) max_pop_per_dimension = 8;
	else max_pop_per_dimension = 6;

	// provjera velicine populacije
	voidP psizep = state->getRegistry()->getEntry("population.size");
	uint popSize = *((uint*)psizep.get());
	if (popSize < max_pop_per_dimension * dimensions) {
		ECF_LOG_ERROR(state,
		"Error: CFO algorithm requires a minimal population size of " +
		uint2str(max_pop_per_dimension * dimensions) + "!");

		throw ("");
	}

	// provjera maksimalnog broja generacija
	voidP maxgenp = state->getRegistry()->getEntry("term.maxgen");
	uint maxGen = *((uint*)maxgenp.get());
	if (maxGen < (max_pop_per_dimension / 2) * 10)
		ECF_LOG(state, 0, "Warning: CFO algorithm requires a minimum of " +
		uint2str((max_pop_per_dimension / 2) * 10) + " generations for optimal work!");

	// parametri iteracije
	pop_per_dimension = 2;	// pocetni broj letjelica po dimenziji
	gamma = 0;				// pocetni nacin stvaranja letjelica (0-9)


	return true;
}


bool CentralForceOptimization::advanceGeneration(StateP state, DemeP deme) {
	// kraj algoritma
	if (pop_per_dimension > max_pop_per_dimension) {
		state->setTerminateCond();
		return false;
	}

	// inicijalizacija granica prostora
	std::vector<double> xmin;
	std::vector<double> xmax;
	for (uint i = 0; i < dimensions; i++) {
		xmin.push_back(xmin_init);
		xmax.push_back(xmax_init);
	}
	
	// stvaranje letjelica ovisno o parametrima iteracije
	std::vector<Probe> probes;	
	spawnProbes(deme, probes, xmin, xmax);
	

	double best_fitness;
	std::vector<double> pos_of_best;
	double fmax_history[25];

	double Frep = Finit;	

	for (uint step = 0; step < 1000; step++) {
		moveProbes(probes);
		retrieveProbes(probes, xmin, xmax, Frep);
		calcProbeMasses(deme, probes);
		calcAccelerations(deme, probes);

		// iteracija faktora vracanja
		Frep += Fdelta;
		if (Frep >= 1) Frep = Fmin;

		findBestFitness(deme, probes, best_fitness, pos_of_best);

		// uvjet ranog izlaska iz petlje
		fmax_history[step % 25] = best_fitness;
		if (step >= 35 && checkTerminationCondition(best_fitness, fmax_history)) break;
		
		// smanjivanje prostora oko najbolje letjelice
		if (step >= 20 && step % 10 == 0) {
			shrinkSearchSpace(xmin, xmax, pos_of_best);
			retrieveProbes(probes, xmin, xmax, Frep);
		}

		// pomocni ispis
		if (best_fitness < global_best) {
			std::cout << "Better solution found at:\n";
			std::cout << " pop per dimension: " << pop_per_dimension << "\n";
			std::cout << " gamma: " << gamma << "\n";
			std::cout << " step: " << step << "\n";
			std::cout << " position:";
			for (uint i = 0; i < dimensions; i++) std::cout << " " << pos_of_best[i];
			std::cout << "\n value: " << best_fitness << "\n\n";

			global_best = best_fitness;
			global_best_pos = pos_of_best;
		}
	}
	
	// za slucaj da se najbolja letjelica pomaknula zbog brzine
	FloatingPointP flp = boost::dynamic_pointer_cast<FloatingPoint::FloatingPoint>( deme->at(0)->getGenotype(0) );
	flp->realValue = global_best_pos;
	evaluate(deme->at(0));


	iterateParameters();

	return true;
}


void CentralForceOptimization::spawnProbes(DemeP deme, std::vector<Probe> &probes, std::vector<double> &xmin, std::vector<double> &xmax) {
	for (uint i = 0; i < dimensions; i++) {					// for each dimension
		for (uint j = 0; j < pop_per_dimension; j++) {		// for each probe in dimension
			
			FloatingPointP flp = boost::dynamic_pointer_cast<FloatingPoint::FloatingPoint>( deme->at(i*pop_per_dimension + j)->getGenotype(0) );
			std::vector<double> &position = flp->realValue;

			for (uint k = 0; k < position.size(); k++) {
				position[k] = xmin[k] + (xmax[k]-xmin[k]) * ((double)gamma/10);					// pozicija na glavnoj dijagonali
			}
			position[i] = xmin[i] + (xmax[i]-xmin[i]) * ((double)j / (pop_per_dimension-1));	// pozicija unutar gledane dimenzije
			
			probes.push_back(Probe(position));
			
		}
	}
}


void CentralForceOptimization::moveProbes(std::vector<Probe> &probes) {
	for (uint i = 0; i < probes.size(); i++) {		// for each probe
		double velocity;
		for (uint j = 0; j < dimensions; j++) {		// for each dimension
			velocity = probes[i].position[j] - probes[i].position_prev[j];
			probes[i].position_prev[j] = probes[i].position[j];
			probes[i].position[j] += velocity + probes[i].acceleration[j] / 2;
		}
	}
}


void CentralForceOptimization::retrieveProbes(std::vector<Probe> &probes, std::vector<double> &xmin, std::vector<double> &xmax, const double Frep) {
	for (uint i = 0; i < probes.size(); i++) {		// for each probe
		bool probe_retrieved = false;
		for (uint j = 0; j < dimensions; j++) {		// for each dimension
			
			if (probes[i].position[j] < xmin[j]) {
				double newpos = xmin[j] + (probes[i].position_prev[j] - xmin[j]) * Frep;
				probes[i].position[j] = (xmin[j] > newpos) ? xmin[j] : newpos;
				probe_retrieved = true;
			}

			else if (probes[i].position[j] > xmax[j]) {
				double newpos = xmax[j] - (xmax[j] - probes[i].position_prev[j]) * Frep;
				probes[i].position[j] = (xmax[j] < newpos) ? xmax[j] : newpos;
				probe_retrieved = true;
			}
			
		}
		// reset position_prev ako je letjelica pomaknuta kako ne bi utjecalo na izracun brzine (veliki pomak)
		if (probe_retrieved) probes[i].position_prev = probes[i].position;
	}
}


void CentralForceOptimization::calcProbeMasses(DemeP deme, std::vector<Probe> &probes) {
	for (uint i = 0; i < probes.size(); i++) evaluate(deme->at(i));
}


void CentralForceOptimization::calcAccelerations(DemeP deme, std::vector<Probe> &probes) {
	for (uint ths = 0; ths < probes.size(); ths++) {			// for each probe
		for (uint i = 0; i < dimensions; i++) probes[ths].acceleration[i] = 0;
		FitnessP mass_ths = deme->at(ths)->fitness;

		for (uint othr = 0; othr < probes.size(); othr++) {		// for each other probe
			FitnessP mass_othr = deme->at(othr)->fitness;
			if (!mass_othr->isBetterThan(mass_ths)) continue;

			// efektivna masa
			double M = abs(mass_othr->getValue() - mass_ths->getValue());

			// udaljenost letjelica
			double abs_distance = 0;
			for (uint i = 0; i < dimensions; i++)
				abs_distance += pow(probes[othr].position[i] - probes[ths].position[i], 2);
			abs_distance = sqrt(abs_distance);

			// racun akceleracije
			for (uint i = 0; i < dimensions; i++) {
				double direction = probes[othr].position[i] - probes[ths].position[i];
				probes[ths].acceleration[i] += M * (direction / pow(abs_distance, 2));
			}
		}
	}
}


void CentralForceOptimization::findBestFitness(DemeP deme, std::vector<Probe> &probes, double &best_fitness, std::vector<double> &pos_of_best) {
	FitnessP best = deme->at(0)->fitness;
	pos_of_best = probes[0].position;
	uint index_of_best = 0;

	for (uint i = 1; i < probes.size(); i++) {
		if (deme->at(i)->fitness->isBetterThan(best)) {
			best = deme->at(i)->fitness;
			pos_of_best = probes[i].position;
			index_of_best = i;
		}
	}

	//probes[index_of_best].position_prev = probes[index_of_best].position;	// reset brzine najbolje letjelice
	best_fitness = best->getValue();
}


bool CentralForceOptimization::checkTerminationCondition(const double best_fitness, double *fmax_history) {
	double avg_fitness = 0;
	for (uint i = 0; i < 25; i++) avg_fitness += fmax_history[i];
	avg_fitness /= 25;

	return (abs(best_fitness - avg_fitness) <= pow(10, -6));
}


void CentralForceOptimization::shrinkSearchSpace(std::vector<double> &xmin, std::vector<double> &xmax, std::vector<double> &pos_of_max) {
	for (uint i = 0; i < dimensions; i++) {
		xmin[i] += (pos_of_max[i] - xmin[i]) / 2;
		xmax[i] -= (xmax[i] - pos_of_max[i]) / 2;
	}
}


void CentralForceOptimization::iterateParameters() {
	gamma++;

	if (gamma > 10) {
		gamma = 0;
		pop_per_dimension += 2;
	}
}